/*
 * Created on 23-Sep-2005
 *
 */
package ssmith.opengl;

/*Title:      mjbWorld
Copyright (c) 1998-2003 Martin John Baker

This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

For information about the GNU General Public License see http://www.gnu.org/

To discuss this program http://sourceforge.net/forum/forum.php?forum_id=122133
   also see website http://www.euclideanspace.com/
   */

/*
   for theory see:
   http://www.euclideanspace.com/maths/algebra/vectors/index.htm
   */

/** This class can represent a 3D vector. For instance a point in 3D space,    or
   a relative position or movement.

The class has methods to add, subtact, (cross and dot) multiply with other
   vectors. also many other methods.
   */
   public class Vector3D {
   	
   	/** VRML only supports float but allow override if higher resolution required    */
   	public static boolean saveAsDouble = false ;
   	/** x coordinate */
   	public double x;
   	/** y coordinate */
   	public double y;
   	/** z coordinate */
   	public double z;
   	
   	/** a constructor to set initial values of x,y and z coodinates
   	 * @param x1 value of x coordinate
   	 * @param y1 value of y coordinate
   	 * @param z1 value of z coordinate
   	 */
   	public Vector3D(double x1,double y1,double z1) {
   		x=x1;
   		y=y1;
   		z=z1;
   	}
   	
   	/** a constructor to set initial values of x,y and z coodinates
   	 * @param x1 value of x coordinate
   	 * @param y1 value of y coordinate
   	 * @param z1 value of z coordinate
   	 */
   	public Vector3D(float x1,float y1,float z1) {
   		x=(double)x1;
   		y=(double)y1;
   		z=(double)z1;
   	}
   	
   	/** copy constructor
   	 * @param in1 set values to save value in1
   	 * returns a new instace with values the same as in1
   	 */
   	public Vector3D(Vector3D in1) {
   		x=(in1!=null) ? in1.x : 0;
   		y= (in1!=null) ? in1.y : 0;
   		z= (in1!=null) ? in1.z : 0;
   	}
   	
   	/** construct a vector with initial value zero. */
   	public Vector3D() {
   	}
   	
   	/** static method to return sum of two vectors
   	 * @param a first vector to be added
   	 * @param b second vector to be added
   	 * @return the sum
   	 */
   	public static Vector3D add(Vector3D a,Vector3D b){
   		return new Vector3D(a.x + b.x,a.y + b.y,a.z + b.z);
   	}
   	
   	/** static method to return difference of two vectors
   	 * @param a first vector
   	 * @param b subract this vector
   	 * @return result
   	 */
   	public static Vector3D sub(Vector3D a,Vector3D b){
   		return new Vector3D(a.x - b.x,a.y - b.y,a.z - b.z);
   	}
   	
   	/** static method to give the VRML name of this vector
   	 */
   	public static String vrmlType_s(){
   		return "SFVec3f";
   	}
   	
   	/** method to give the VRML name of this vector
   	 sometimes we cant use vrmlType_s because static methods cant be overriden
   	 */
   	public String vrmlType(){
   		return "SFVec3f";
   	}
   	
   	/** overrides base clone method
   	 * @return new instance with same value of this
   	 */
   	public Object clone() {
   		return new Vector3D(this);
   	}
   	
   	/** set the value of this instance to the value of other
   	 * this can be used to reuse an instance without the overheads of garbidge collection
   	 * @param other instace we want to use value of, if null then set to zero
   	 */
   	public void copy(Vector3D other){
   		if (other==null) {
   			x=y=z=0;
   			return;
   		}
   		x= other.x;
   		y= other.y;
   		z= other.z;
   	}
   	
   	/** subtract other vector from this
   	 * for theory see:
   	 * http://www.euclideanspace.com/maths/algebra/vectors/index.htm
   	 * @param other vector to be subtracted from this
   	 */
   	public void sub(Vector3D other){
   		if (other==null) return;
   		x-= other.x;
   		y-= other.y;
   		z-= other.z;
   	}
   	
   	/** add other vector to this
   	 * for theory see:
   	 * http://www.euclideanspace.com/maths/algebra/vectors/index.htm
   	 * @param other vector to be added to this
   	 */
   	public void add(Vector3D other){
   		if (other==null) return;
   		x+= other.x;
   		y+= other.y;
   		z+= other.z;
   	}
   	
   	/** inverts the direction of the vector */
   	public void minus() {
   		x=-x; y=-y; z=-z;
   	}
   	
   	/** return a vector pointing on the opposite direction to this without affecting    the
   	 * value of this instance
   	 * @return new instance with value= munus this
   	 */
   	public Vector3D getMinus() {
   		return new Vector3D(-x,-y,-z);
   	}
   	
   	/** convert this vector to unit length
   	 * for theory see:
   	 * http://www.euclideanspace.com/maths/algebra/vectors/normals/index.htm
   	 */
   	public void normalize(){
   		double t = Math.sqrt(x*x + y*y + z*z);
   		x /= t;
   		y /= t;
   		z /= t;
   	}
   	
   	/** scale this vector equally in all directions
   	 * @param sc scale factor
   	 */
   	public void scale(double sc){
   		x *= sc;
   		y *= sc;
   		z *= sc;
   	}
   	
   	/** scale this vector posibly different in x,y and z directions
   	 * @param other scale value
   	 */
   	public void scale(Vector3D other){
   		x *= other.x;
   		y *= other.y;
   		z *= other.z;
   	}
   	
   	/** scale this vector by inverse of other
   	 * @param other scale value
   	 */
   	public void scaleInv(Vector3D other){
   		x /= other.x;
   		y /= other.y;
   		z /= other.z;
   	}
   	
   	/** return square of distance from end of this vector to end of other
   	 * @param other calcules distance from this vector
   	 * @return square of distance from end of this vector to end of other
   	 */
   	public double distanceSquared(Vector3D other){
   		double x1 = other.x - x;
   		double y1 = other.y - y;
   		double z1 = other.z - z;
   		return x1*x1 + y1*y1 + z1*z1;
   	}
   	
   	/** cross product
   	 * for theory see:
   	 * http://www.euclideanspace.com/maths/algebra/vectors/index.htm
   	 * @param other vector to take cross product with
   	 */
   	public void cross(Vector3D other) {
   		double xh = y * other.z - other.y * z;
   		double yh = z * other.x - other.z * x;
   		double zh = x * other.y - other.x * y;
   		x = xh;
   		y = yh;
   		z = zh;
   	}
   	
   	/** dot product
   	 * for theory see:
   	 * http://www.euclideanspace.com/maths/algebra/vectors/index.htm
   	 * @param other vector to take dot product with
   	 * @return dot product
   	 */
   	public double dot(Vector3D other) {
   		return (x * other.x) + (y * other.y) + (z * other.z);
   	}
   	
   	/** set the x value only without althering the other dimentions
   	 * @param v new value for x
   	 */
   	public void setX(double v) {
   		x=v;
   	}
   	
   	/** set the y value only without althering the other dimentions
   	 * @param v new value for y
   	 */
   	public void setY(double v) {
   		y=v;
   	}
   	
   	/** set the z value only without althering the other dimentions
   	 * @param v new value for z
   	 */
   	public void setZ(double v) {
   		z=v;
   	}
   	
   	/** gets the value in the x dimension
   	 * @return the value in the x dimension
   	 */
   	public double getx() {
   		return x;
   	}
   	
   	/** gets the value in the y dimension
   	 * @return the value in the y dimension
   	 */
   	public double gety() {
   		return y;
   	}
   	
   	/** gets the value in the z dimension
   	 * @return the value in the z dimension
   	 */
   	public double getz() {
   		return z;
   	}
   	
   	/** return a string which represents the value which can be used in source    code
   	 * @return a string representing the value of this
   	 */
   	public String toStatic() {
   		return ""+ x + "f," + y + "f," + z + "f";
   	}
   	
   	/** returns true if any dimension of other vector is greater than the same    dimension
   	 * of this
   	 * @param other vector to compare with this
   	 * @return true if greater
   	 */
   	public boolean greaterThan(Vector3D other) {
   		if (other.x > x) return true;
   		if (other.y > y) return true;
   		if (other.z > z) return true;
   		return false;
   	}
   	
   	/** returns true if any dimension of other vector is less than the same dimension
   	 * of this
   	 * @param other vector to compare with this
   	 * @return true if less
   	 */
   	public boolean lessThan(Vector3D other) {
   		if (other.x < x) return true;
   		if (other.y < y) return true;
   		if (other.z < z) return true;
   		return false;
   	}
   	
   	/** returns true if this vector has an equal value to other vector
   	 * @param other vector to compare with this
   	 * @return
   	 */
   	public boolean equals(Vector3D other) {
   		if (other==null) return false;
   		if (x!= other.x) return false;
   		if (y!= other.y) return false;
   		if (z!= other.z) return false;
   		return true;
   	}
   	
   	/** output as a string
   	 * @param format">mode values
   	 * 0 - output modified values
   	 * 1 - output original values
   	 * 2 - output attribute
   	 * 3 - output attribute in brackets
   	 * 4 - output with f prefix
   	 * @return string result
   	 */
   	public String outstring(int format) {
   		if (format == 3) {
   			if (saveAsDouble)
   				return "(" + x + " " + y + " " + z + ")";
   			else
   				return "(" + new Float(x).toString() + " " +
				new Float(y).toString() + " " +
				new Float(z).toString() + ")";
   		} else if (format == 4) {
   			return new Float(x).toString() + "f," +
			new Float(y).toString() + "f," +
			new Float(z).toString() + "f";
   		} else {
   			if (saveAsDouble)
   				return "" + x + " " + y + " " + z;
   			else
   				return new Float(x).toString() + " " +
				new Float(y).toString() + " " +
				new Float(z).toString();
   		}
   	}
   	
   	
   	/** use to combine bounds
   	 * if i=0 take minimum otherwise maximum
   	 * @param other">vector to combine with</param>
   	 * @param i">if i=0 take minimum otherwise maximum</param>
   	 * */
   	public void bounds(Vector3D other,int i){
   		if (i==0) { // take minimum
   			if (other.x < x) x=other.x;
   			if (other.y < y) y=other.y;
   			if (other.z < z) z=other.z;
   		} else { // take maximum
   			if (other.x > x) x=other.x;
   			if (other.y > y) y=other.y;
   			if (other.z > z) z=other.z;
   		}
   	}
   	

   }