package ssmith.opengl;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.StringTokenizer;
import ssmith.lang.Functions;

/**
 * The data that has been read from the Wavefront .obj file. This is
 * kept seperate from the actual rendering with the hope the data
 * might be used for some other rendering engine in the future.
 *
 * @author Kevin Glass
 */
public class ModelData {

        /** The faces data read from the file */
        private ArrayList groups = new ArrayList();

        /** The verticies that have been read from the file */
        private ArrayList verts = new ArrayList();
        /** The texture coordinates that have been read from the file */
        private ArrayList texCoords = new ArrayList();
        /** The normals that have been read from the file */
        private ArrayList normals = new ArrayList();

        /**
         * Create a new set of OBJ data by reading it in from the specified
         * input stream.
         *
         * @param in The input stream from which to read the OBJ data
         * @throws IOException Indicates a failure to read from the stream
         */
        public ModelData(InputStream in, float scale) throws IOException {
            // read the file line by line adding the data to the appropriate
            // list held locally

            ArrayList faces = new ArrayList();
            int l = 0;

            BufferedReader reader = new BufferedReader(new InputStreamReader(in));

            while (reader.ready()) {
                String line = reader.readLine();
                l++;
                //System.out.println("Read line " + l);

                // if we read a null line thats means on some systems
                // we've reached the end of the file, hence we want to
                // to jump out of the loop
                if (line == null) {
                    break;
                }

                // "vn" indicates normal data
                if (line.startsWith("vn")) {
                    Tuple3 normal = readTuple3(line);
                    normals.add(normal);
                // "vt" indicates texture coordinate data
                } else if (line.startsWith("vt")) {
                    Tuple3 tex = readTuple3(line);
                    texCoords.add(tex);
                // "v" indicates vertex data
                } else if (line.startsWith("v")) {
                    Tuple3 vert = readTuple3(line);
                    vert.scale(scale);
                    verts.add(vert);
                // "f" indicates a face
                } else if (line.startsWith("f")) {
                    Face face = readFace(line);
                    faces.add(face);
                } else if (line.startsWith("g")) { // Next group
                    groups.add(faces);
                    faces = new ArrayList();
                }
            }

            // Print some diagnositics data so we can see whats happening
            // while testing
            /*if (Client.SHOW_MODEL_LOADING) {
                System.out.println("Read " + groups.size() + " groups");
                System.out.println("Read " + verts.size() + " verticies");
            }*/
        }

        /**
         * Get the number of faces found in the model file
         *
         * @return The number of faces found in the model file
         */
        public int getFaceCount(int grp) {
            ArrayList faces = (ArrayList)groups.get(grp);
            return faces.size();
        }

        /**
         * Get the data for specific face
         *
         * @param index The index of the face whose data should be retrieved
         * @return The face data requested
         */
        public Face getFace(int grp, int index) {
            ArrayList faces = (ArrayList)groups.get(grp);
            return (Face) faces.get(index);
        }

        public int getGroupCount() {
            return groups.size();
        }

        /**
         * Read a set of 3 float values from a line assuming the first token
         * on the line is the identifier
         *
         * @param line The line from which to read the 3 values
         * @return The set of 3 floating point values read
         * @throws IOException Indicates a failure to process the line
         */
        private Tuple3 readTuple3(String line) throws IOException {
            StringTokenizer tokens = new StringTokenizer(line, " ");

            tokens.nextToken();

            try {
                float x = Float.parseFloat(tokens.nextToken());
                float y = Float.parseFloat(tokens.nextToken());
                float z = 0;
                if (tokens.hasMoreTokens()) {
                    z = Float.parseFloat(tokens.nextToken());
                }

                return new Tuple3(x,y,z);
            } catch (NumberFormatException e) {
                throw new IOException(e.getMessage());
            }
        }

        /**
         * Read a set of 2 float values from a line assuming the first token
         * on the line is the identifier
         *
         * @param line The line from which to read the 3 values
         * @return The set of 2 floating point values read
         * @throws IOException Indicates a failure to process the line
         */
        private Tuple2 readTuple2(String line) throws IOException {
            StringTokenizer tokens = new StringTokenizer(line, " ");

            tokens.nextToken();

            try {
                float x = Float.parseFloat(tokens.nextToken());
                float y = Float.parseFloat(tokens.nextToken());

                return new Tuple2(x,y);
            } catch (NumberFormatException e) {
                throw new IOException(e.getMessage());
            }
        }

        /**
         * Read a set of face data from the line
         *
         * @param line The line which to interpret as face data
         * @return The face data extracted from the line
         * @throws IOException Indicates a failure to process the line
         */
        private Face readFace(String line) throws IOException {
            StringTokenizer points = new StringTokenizer(line, " ");

            points.nextToken();
            int faceCount = points.countTokens();

            // currently we only support triangels so anything other than
            // 3 verticies is invalid

            if (faceCount != 3) {
                throw new RuntimeException("Only triangles are supported");
            }

            // create a new face data to populate with the values from the line
            Face face = new Face(faceCount);

            try {
                // for each line we're going to read 3 bits of data, the index
                // of the vertex, the index of the texture coordinate and the
                // normal.

                for (int i=0;i<faceCount;i++) {
                    StringTokenizer parts = new StringTokenizer(points.nextToken(), "/");

                    int v = Integer.parseInt(parts.nextToken());
                    //v = Functions.mod(v);
                    if (v < 0) {
                        v = verts.size()+v;
                    } else {
                        v--;
                    }

                    int t = Integer.parseInt(parts.nextToken());
                    if (t < 0) {
                        t = texCoords.size()+t;
                    } else {
                        t--;
                    }
                    //t = Functions.mod(t);
                    /*int n = 0;
                    if (parts.hasMoreTokens()) {
                        n = Integer.parseInt(parts.nextToken());
                    }*/

                    // we have the indicies we can now just add the point
                    // data to the face.
    /*                face.addPoint((Tuple3) verts.get(v-1),
                            (Tuple3) texCoords.get(t-1),
                            (Tuple3) normals.get(n-1));*/
                    try {
    /*                    System.out.println("v=" + v);
                        System.out.println("verts.get(v)=" + verts.get(v));
                        System.out.println("t=" + t);
                        System.out.println("texCoords.get(t)=" + texCoords.get(t));
      */
                        Tuple3 tmp = null;
                        if (texCoords.size() > t) {
                            tmp = (Tuple3)texCoords.get(t);
                        } else {
                            tmp = new Tuple3(Functions.rndFloat(0, 1), Functions.rndFloat(0, 1), Functions.rndFloat(0, 1));
                        }
                        face.addPoint((Tuple3) verts.get(v),
                                      (Tuple3) tmp,
                                      new Tuple3(0f, 1f, 0f));
                    } catch (IndexOutOfBoundsException e) {
                            e.printStackTrace();
                        }
                }
            } catch (NumberFormatException e) {
                throw new IOException(e.getMessage());
            }

            return face;
        }

    }


