package ssmith.opengl;

import ssmith.lang.*;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.ListIterator;

import net.java.games.jogl.*;
import net.java.games.jogl.util.*;
import ngrave.client.Client;

/**
 *
 * Notes about openGL: It only seems to like colour if you specify the rgb
 * explicitly as floats.
 * List numbers must be 1 or greater Using lists doesn't
 * seem to make any difference.
 * Binding textures must be before you specify
 * QUADs etc...
 * Do not look directly from above (for some reason).
 *
 */
public class Canvas3D implements GLEventListener {

  public static boolean texture_on = true;
  public static boolean global_lights_on = false;
  public static boolean draw_gl = true; // Set to false to disable OpenGL drawing

  public GLCanvas glcanvas;
  public static Textures textures;
  public static GLUquadric quadratic; // for use by all objects
  private ICamera camera;
  public static boolean is_drawing = false; // indicates when started
  private Animator anim;
  private ArrayList objects = new ArrayList();
  private float max_view_dist;
  public static long draw_time;
  public ArrayList effects = new ArrayList();
  private GLUT glut = new GLUT();
  private int canvas_width, canvas_height;
  private IManualDraw mdraw;
  private float height_ratio;
  private boolean change_max_view_dist_on_next_redraw = false;
  private IGameLoop game_loop;
  private boolean textures_loaded = false;
  private int loop_delay;

  public Canvas3D(float view_dist, ICamera cam, IManualDraw md, IGameLoop g_loop, Object listen, int l_delay) {
  	max_view_dist = view_dist;
  	this.mdraw = md;
  	//this.textwriter = writer;
  	this.game_loop = g_loop;
    this.loop_delay = l_delay;

  	glcanvas = getGLCanvas();
    glcanvas.addGLEventListener(this);
    glcanvas.setFocusable(false);
    anim = new Animator(glcanvas);
    camera = cam;

    if (cam == null) {
    	System.err.println("Warning: Camera is null!");
    }
  }

  public void setTextures(Textures tex) {
      textures = tex;
  }

  public void init(GLDrawable drawable) {
    final GL gl = drawable.getGL();
    final GLU glu = drawable.getGLU();

    if (textures != null) {
    	textures.setGL(gl, drawable);
    } else {
        System.out.println("Canvas3D Warning - no texture loader set.");
    }

    gl.glShadeModel(GL.GL_SMOOTH);
    gl.glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
    gl.glClearDepth(1.0f); // Depth Buffer Setup
    gl.glEnable(GL.GL_DEPTH_TEST); //Enables Depth Testing
    gl.glDepthFunc(GL.GL_LEQUAL); // The Type Of Depth Testing To Do
    gl.glBlendFunc(GL.GL_SRC_ALPHA, GL.GL_ONE);

    // New
    gl.glHint(GL.GL_PERSPECTIVE_CORRECTION_HINT, GL.GL_NICEST);  // Really Nice Perspective Calculations

    for(int i=0; i<effects.size() ; i++) {
        AbstractEffect effect = (AbstractEffect)effects.get(i);
        effect.init(this, gl);
    }

    quadratic = glu.gluNewQuadric(); // Create A Pointer To The Quadric Object
    glu.gluQuadricNormals(quadratic, GLU.GLU_SMOOTH); // Create Smooth Normals
    glu.gluQuadricTexture(quadratic, true); // Create Texture Coords

    is_drawing = true;
  }

  public void startAnim() {
      this.anim.start();
  }

  public void stopAnim() {
      this.anim.stop();
  }

  public void setMaxViewDist(float dist) {
  	max_view_dist = dist;
  	change_max_view_dist_on_next_redraw = true;
  }

  public float getMaxViewDist() {
      return max_view_dist;
  }

  public void reshape(GLDrawable drawable, int x, int y, int width, int height) {
    is_drawing = false;

    this.canvas_width = width;
    this.canvas_height = height;

    final GL gl = drawable.getGL();
    final GLU glu = drawable.getGLU();

    quadratic = glu.gluNewQuadric(); // Create A Pointer To The Quadric Object
    glu.gluQuadricNormals(quadratic, GLU.GLU_SMOOTH); // Create Smooth Normals
    glu.gluQuadricTexture(quadratic, true); // Create Texture Coords

    if (height <= 0) {// avoid a divide by zero error!
      height = 1;
    }
    height_ratio = (float) width / (float) height;
    gl.glViewport(0, 0, width, height);
    gl.glMatrixMode(GL.GL_PROJECTION);
    gl.glLoadIdentity();
    glu.gluPerspective(45.0f, height_ratio, 0.1, this.max_view_dist); // seems to be view_angle, ratio, camera_dist, max_view_dist;
    gl.glMatrixMode(GL.GL_MODELVIEW);
    gl.glLoadIdentity();

    is_drawing = true;
  }
  
  public void loadTextures() {
      if (textures_loaded == false) {
      if (textures != null) {
          textures.loadTextures();
          textures_loaded = true;
      }
  }
  }

  public void display(GLDrawable drawable) {
	  if (game_loop != null) {
		  game_loop.glGameLoop();
	  }

      if (draw_gl) {
          long start = System.currentTimeMillis();

          GL gl = drawable.getGL();
          GLU glu = drawable.getGLU();

          if (change_max_view_dist_on_next_redraw) {
            gl.glViewport(0, 0, this.canvas_width, this.canvas_height);
            gl.glMatrixMode(GL.GL_PROJECTION);
            gl.glLoadIdentity(); // 5 was 45
            glu.gluPerspective(45.0f, height_ratio, 0.001, max_view_dist); // seems to be view_angle, ratio, camera_dist, max_view_dist;
            gl.glMatrixMode(GL.GL_MODELVIEW);
            gl.glLoadIdentity();

            change_max_view_dist_on_next_redraw = false;
          }

          gl.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT);
          gl.glLoadIdentity();

          gl.glCullFace(GL.GL_BACK);
          gl.glEnable(GL.GL_CULL_FACE);

          if (texture_on) {
              gl.glEnable(GL.GL_TEXTURE_2D);
          }
          else {
              gl.glDisable(GL.GL_TEXTURE_2D);
          }

          AbstractEffect effect;
          for(int i=0; i<effects.size() ; i++) {
              effect = (AbstractEffect)effects.get(i);
              effect.process(gl);
          }

          if (Canvas3D.global_lights_on) {
              gl.glEnable(GL.GL_LIGHTING);
          } else {
              gl.glDisable(GL.GL_LIGHTING);
          }

          /*      if (Canvas3D.fog_on) {
           gl.glEnable(GL.GL_FOG);
           } else {
           gl.glDisable(GL.GL_FOG);
           }*/

          try {
              Vector from = camera.getViewFrom();
              Vector to = camera.getViewTo();
              Vector or = camera.getOrientation();
              glu.gluLookAt(from.x, from.y, from.z, to.x, to.y, to.z, or.x, or.y, or.z);
              //glu.gluLookAt(10f, 10f, 10f, 0, 0, 0, 0f, 1f, 0f);

              //draw objects
              I3DObject obj;
              ListIterator it = objects.listIterator();
              while (it.hasNext()) {
                      obj = (I3DObject)it.next();
                      if (obj.shouldBeDrawn()) {
                          gl.glPushMatrix();
                          obj.draw(gl, glu);
                          gl.glPopMatrix();
                      }
              }

              // Manual drawer
              if (this.mdraw != null) {
              	this.mdraw.drawGL(gl, glu);
              }

              // Text
              /*if (this.textwriter != null) {
	              this.startTextMode(gl, glu);
	              this.textwriter.writeText(gl);
	              this.finishTextMode(gl);
              }*/
          }

          catch (ConcurrentModificationException ex) {
              System.err.println(ex.getMessage());
          }
          catch (Exception ex) {
              System.err.println(ex.getMessage());
              ex.printStackTrace();
              System.exit(-1);
          }

          draw_time = System.currentTimeMillis() - start;
          long wait = loop_delay - draw_time;//System.currentTimeMillis() + start;
          if (Client.DEBUG) {
           if (wait < 0) {
            System.out.println("Client too slow! " + wait);
           }
               }
          Functions.delay(wait);
      }
  }

  public void displayChanged(GLDrawable drawable, boolean modeChanged,
                             boolean deviceChanged) {
    // Not implemented by JOGL.
  }

  private GLCanvas getGLCanvas() {
    GLCapabilities capabilities = new GLCapabilities();
    return GLDrawableFactory.getFactory().createGLCanvas(capabilities);
  }

  public void startTextMode(GL gl, GLU glu) {
      // switch to projection mode
      gl.glMatrixMode(GL.GL_PROJECTION);
      // save previous matrix which contains the
      //settings for the perspective projection
      gl.glPushMatrix();
      // reset matrix
      gl.glLoadIdentity();
      // set a 2D orthographic projection
      glu.gluOrtho2D(0, this.canvas_width, 0, this.canvas_height);
      // invert the y axis, down is positive
      gl.glScalef(1, -1, 1);
      // mover the origin from the bottom left corner
      // to the upper left corner
      gl.glTranslatef(0, -this.canvas_height, 0);
      gl.glMatrixMode(GL.GL_MODELVIEW);

      gl.glPushMatrix();
      gl.glLoadIdentity();
      gl.glDisable(GL.GL_TEXTURE_2D);
  }

  public void writeString(GL gl, int x, int y, int font, String string) {
      for (int c=0; c < string.length(); c++) {
          gl.glRasterPos2f(x,y);
          char ch = string.charAt(c);
          glut.glutBitmapCharacter(gl, font, ch);
          x = x + glut.glutBitmapWidth(font, ch);
      }
  }

  public void addObject(I3DObject obj) {
      this.objects.add(obj);
  }

  public void removeObject(I3DObject obj) {
      this.objects.remove(obj);
  }

  public int getHeight() {
	  return this.canvas_height;
  }

  public int getWidth() {
	  return this.canvas_width;
  }

  public void finishTextMode(GL gl) {
      gl.glPopMatrix();
      gl.glMatrixMode(GL.GL_PROJECTION);
      gl.glPopMatrix();
      gl.glMatrixMode(GL.GL_MODELVIEW);
  }

  public void removeAllObjects() {
      objects.clear();
  }

  public static void SetRandomColour(GL gl) {
      gl.glColor3f(Functions.rndFloat(0, 1f), Functions.rndFloat(0, 1f), Functions.rndFloat(0, 1f));
  }

}
