package ngrave.server.objects;

import ssmith.io.TextFile;
import ssmith.lang.Functions;
import java.util.*;
import java.io.*;

import ngrave.ai.AI;
import ngrave.client.Client;
import ngrave.server.CollisionDetails;
import ngrave.server.IDamagable;
import ngrave.server.S2CCommunication;
import ngrave.server.SMapSquare;
import ngrave.server.SPlayerData;
import ngrave.server.Server;
import ngrave.server.Statics;
import ngrave.server.items.SItemCarried;
import ngrave.server.items.SWeaponCarried;
import ngrave.shared.PlayerObject3D;

public abstract class SPlayerObject3D extends SObject implements IDamagable {

    // Action point costs
    public static final int OPEN_DOOR_COST=5, TURN_OFF_COMPUTER_COST=10;
    public static final int CHANGE_ITEM_COST=10, RELOAD_COST=19;
    public static final float MOVE_FWD_COST=4f;

    // Shot types
    public static final int SNAP_SHOT=0, NORMAL_SHOT=1, AIMED_SHOT=2; // Must be the same order as the combo box in which they are selected!

    // Stats
    private float accuracy;
    private int health, max_health;
    public int armour_damage_reduction; //todo - make private
    private int combat, shot_type = NORMAL_SHOT;

    private float action_points = Server.MAX_ACTION_POINTS;
    public final float move_fwd_cost; //todo
    public String name = "None set";
    private SPlayerObject3D can_see_enemy, prev_can_see_enemy;
    private boolean autofire = Client.DEFAULT_AUTOFIRE, guarding = false;
    private final boolean is_valid_target = true;
    private ArrayList items = new ArrayList();
    protected SItemCarried current_item;
    protected AI ai;
    private long action_time; // The time at which the trooper can do something
    private CollisionDetails no_move_reason = new CollisionDetails();
    private static ArrayList firstnames, surnames;
    private static final long CAN_SEE_ENEMY_DELAY = 500;
    private long last_time_looked_for_enemy = 0;
    private SPlayerData current_controller;

    static {
        loadNames();
    }

    public SPlayerObject3D(float x, float z, float diam, byte side, float move_cost, String name, int health, boolean equip, int combat_power, float acc) {
        super(x, z, 0, diam, side, false, Client.MOVE_SPEED);
        this.move_fwd_cost = move_cost;
        this.name = name;
        this.health = health;
        this.max_health = health;
        //this.equipable = equip;
        this.combat = combat_power;
        this.accuracy = acc;

        sendAllDetails();

        // Line us up in the middle of the square;
        int x2 = (int)getX();
        int z2 = (int)getZ();
        this.setLocation(x2 + 0.5f, z2 + 0.5f, false);

        // Point us towards the centre
        int ang = this.getAbsoluteAngleTo(Server.map.size/2, Server.map.size/2);
        this.turnRegardless(ang, true, true, true);

        //SPlayerData pdata = ServerFunctions.GetPlayerDataForSide(owner);
        /*pdata.troopers++;
        if (pdata.isCPU()) {
            this.computer_controlled = true;
        }*/

        Server.game.addUnitToSide(this);
    }

    public boolean isComputerControlled() {
        return this.ai != null;
    }

    public void sendAllDetails() {
        S2CCommunication.SendNewObjectToAll(this);
        //S2CCommunication.SendTroopersItemsToAll(this); No point since they won't have any yet!
        S2CCommunication.SendHealthLevelToAll(this);
        S2CCommunication.SendArmourLevelToAll(this);
        S2CCommunication.SendTrooperStatsToAll(this);

        // See if any players can pre-select us
        Iterator it = Server.players.getIterator();
        while (it.hasNext()) {
            SPlayerData pdata = (SPlayerData)it.next();
            if (pdata.side == this.side) {
                if (pdata.getCurrentUnit() == null) {
                    pdata.selectUnit(this);
                }
            }
        }
    }

    public void setAI(AI ai) {
        if (this.ai == null) {
            Server.game.unitIsNowCPU(this);
        }
        this.ai = ai;
    }

    public void process() {
        if (this.isComputerControlled()) {
            ai.doAI();
        } else {
            SPlayerObject3D obj = this.findAnEnemy();
            this.can_see_enemy = obj;
            // Can we now see an enemy, and we couldn't before?  (Or vice-versa?)
            if (this.can_see_enemy != this.prev_can_see_enemy) {
                S2CCommunication.SendCanSeeEnemy(this);
                this.prev_can_see_enemy = this.can_see_enemy;
            }
            if (this.can_see_enemy != null) {
                this.setGuard(false, true); // Turn off skip
                if (this.autofire) {
                    this.reload(true); // Just in case
                    this.turnRegardless(this.getAbsoluteAngleTo(obj), true, true, true);
                    // Check we're not being controlled
                    if (this.current_controller == null) {
                        if (this.current_item != null) {
                            if (this.current_item instanceof SWeaponCarried) {
                                SWeaponCarried wep = (SWeaponCarried)this.current_item;
                                if (wep.can_autofire) {
                                    if (this.isAngleClearOfComrades(this.distanceTo(obj))) {
                                        this.shoot(); // Shoot if not being controlled
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

    }

    public SItemCarried getCurrentItem() {
        return this.current_item;
    }

    public boolean hasAnyItems() {
        return this.items.size() > 0;
    }

    public void setCurrentItemByNo(int no) {
        this.current_item = this.getItemByRef(no);
    }

    public boolean getCanSeeEnemy() {
        return this.can_see_enemy != null;
    }

    public SItemCarried getItemByRef(int no) {
        if (no < 0) {
            return null;
        }
        SItemCarried item;
        for (int i=0 ; i<items.size() ; i++) {
            item = (SItemCarried) items.get(i);
            if (item.ref == no) {
                return item;
            }
        }
        return null;
    }

    public int getNoOfItemsCarried() {
        return this.items.size();
    }

    public int getArmourLevel() {
        return this.armour_damage_reduction;
    }

    public void reduceArmour(int amt) {
        this.armour_damage_reduction -= amt;
        if (this.armour_damage_reduction < 0) {
            this.armour_damage_reduction = 0;
        }
        S2CCommunication.SendArmourLevelToAll(this);
    }

    public void setArmour(int amt) {
        this.armour_damage_reduction = amt;
        S2CCommunication.SendArmourLevelToAll(this);
    }

    public int getHealth() {
        return this.health;
    }

    public void reduceHealth(int amt) {
        this.health -= amt;
        if (health < 0) {
            health = 0;
        }
        S2CCommunication.SendHealthLevelToAll(this);
        if (this.health <= 0) {
            this.remove(true);
        }
    }

    public void heal() {
        this.health = this.max_health;
        S2CCommunication.SendHealthLevelToAll(this);
    }

    public SItemCarried getItemByPos(int no) {
        return (SItemCarried) items.get(no);
    }

    public void addItem(SItemCarried item) {
        this.items.add(item);
        if (this.current_item == null) {
            this.current_item = item;
        }
        S2CCommunication.SendTroopersItemsToAll(this);
    }

    public void removeItem(int no) {
        SItemCarried item;
        for (int i=0 ; i<items.size() ; i++) {
            item = (SItemCarried) items.get(i);
            if (item.ref == no) {
                removeItem(item);
            }
        }
    }

    public void removeItem(SItemCarried item) {
        S2CCommunication.SendRemoveItem(this, item);
        items.remove(item);
        if (this.current_item == item) {
            this.current_item = null;
        }
    }

    /**
     * This checks there are no friends within our angle of fire.
     */
    public boolean isAngleClearOfComrades(float dist_to_enemy) {
        SPlayerObject3D obj;
        for(int o=0 ; o<Server.objects.length ; o++) {
            if (Server.objects[o] != null) {
                if (Server.objects[o] instanceof SPlayerObject3D) {
                    obj = (SPlayerObject3D) Server.objects[o];
                    if (obj.side == this.side) {
                        if (obj != this) {
                            float dist = this.distanceTo(obj);
                            if (dist < dist_to_enemy) { // Check we're not looking at a friend behind a wall or something.
                                int ang = this.getAbsoluteAngleTo(obj);
                                int max_ang = 7;
                                if (dist < 2) {
                                    max_ang = 20;
                                } else if (dist < 1) {
                                    max_ang = 40;
                                }
                                int diff = Functions.GetDiffBetweenAngles(this.getAngle(), ang);
                                if (diff <= max_ang) {
                                    //System.out.println("Shooter angle: " + this.y_angle + "  Target angle: "+ang);
                                    return false;
                                }
                            }
                        }
                    }
                }
            }
        }
        return true;
    }

    /**
     * This ensures they will actively look for the enemy next process(), otherwise
     * they might not since they only check every few seconds.
     *
     */
    public void checkForEnemyNextTime() {
        last_time_looked_for_enemy = 0;
    }

    public SPlayerObject3D findAnEnemy() {
        // Only check for a new enemy every CAN_SEE_ENEMY_DELAY ms.
        if (System.currentTimeMillis() -
                last_time_looked_for_enemy < CAN_SEE_ENEMY_DELAY) {
            return this.prev_can_see_enemy;
        } else {
            //System.out.println("Looking for enemy...");
            last_time_looked_for_enemy = System.currentTimeMillis();
        }
        // See if they can see an enemy
        SObject obj;
        SObject closest = null;
        float max_dist = 9999f;
        for (int o = 0; o < Server.objects.length; o++) {
            obj = Server.objects[o];
            if (obj != null) {
                // See if they can see an enemy
                if (obj instanceof SPlayerObject3D) {
                    SPlayerObject3D trooper = (SPlayerObject3D) obj;
                    if (trooper.is_valid_target) {
                        if (obj.side != this.side) {
                            if (this.canSee(obj, true)) {
                                float dist = this.distanceTo(obj);
                                if (dist < max_dist) {
                                    closest = obj;
                                    max_dist = dist;
                                }
                            }
                        }
                    }
                }
            }

        }
        return (SPlayerObject3D) closest;
    }

    public boolean canTraverse(SMapSquare sq) {
        return sq.blocks_movement == false; // < SMapSquare.START_OF_VIEW_BLOCKERS;
    }

    public boolean hasCollidedWith(SObject other) {
        if (other instanceof SPlayerObject3D) {
            SPlayerObject3D opponent = (SPlayerObject3D) other;
            if (other.side != this.side) {
                // Close combat!
                int att = Functions.rnd(this.combat / 2, this.combat);
                int def = Functions.rnd(opponent.combat / 2, opponent.combat);
                System.out.println("Close combat between " + this + " and " + other);
                System.out.println("Att: " + att + "  def: " + def);
                att -= def;
                if (att > 0) {
                    opponent.damage(att, this.getAbsoluteAngleTo(opponent));
                }
            }
            return true;
        }
        return false;
    }

    public SMapSquare getMapSquare() {
        return Server.map.getMapSquare(this.getX(), this.getZ());
    }

    /**
     * There is the equivalent function is CPlayerObject3D
     * @param cost
     * @return
     */
/*    public float adjustCostForArmour(float cost) {
        return cost * (100 + (armour_damage_reduction*2)) / 100;
    }*/

    /**
     * This needs to return the reason they can't move for the AI.
     * @return
     */
    public CollisionDetails moveForwardsIfEnoughAPs() {
        //System.out.println(name + " trying to move fwds.");
        if (System.currentTimeMillis() > this.action_time) {
            float cost = move_fwd_cost * Client.MOVE_SPEED;
            cost = PlayerObject3D.adjustCostForArmour(this.armour_damage_reduction, cost);
            if (this.getAPs() >= cost) {
                if (attemptMoveForwards().moveSuccessful()) {
                    S2CCommunication.SendAnimate(this);
                    this.reduceAPs(cost, true);
                    if (isDestroyed() == false) { // in case we've been removed (i.e. an exit)
                        Server.game.mission.trooperEnteredSquare(this, this.getMapSquare());
                    }
                }
            }
        }
        return this.no_move_reason;
    }

        public void moveBackwardsIfEnoughAPs() {
        if (System.currentTimeMillis() > this.action_time) {
            float cost = move_fwd_cost * Client.MOVE_SPEED;
            cost = PlayerObject3D.adjustCostForArmour(this.armour_damage_reduction, cost);
            if (this.getAPs() >= cost) {
                if (attemptMoveBackwards()) {
                    this.reduceAPs(cost, true);
                    Server.game.mission.trooperEnteredSquare(this, this.getMapSquare());
                } else {
                    return;// false; // So the AI knows its hit a wall.
                }
                return;// true;
            }
        }
        return;// false;
    }

    public void turnToIfEnoughAPs(SMapSquare ms) {
        this.turnIfEnoughAPs(this.getAbsoluteAngleTo(ms), true);
    }

    public boolean turnTowardsIfEnoughAPs(SMapSquare ms) {
        return this.turnTowardsIfEnoughAPs(this.getAbsoluteAngleTo(ms));
    }

    public boolean turnTowardsIfEnoughAPs(int angle) {
        final int turn_interval = Client.TURN_SPEED;

        if (Functions.GetDiffBetweenAngles(this.getAngle(), angle) <= turn_interval) {
            return this.turnIfEnoughAPs(angle, true);
        } else {
            int angle_to_turn = angle - this.getAngle();
            while (angle_to_turn < -180) {
                angle_to_turn += 360;
            }
            while (angle_to_turn > 180) {
                angle_to_turn -= 360;
            }
            if (angle_to_turn > 0) {
                if (angle_to_turn > turn_interval) {
                    angle_to_turn = turn_interval;
                }
            } else {
                if (angle_to_turn < turn_interval) {
                    angle_to_turn = -turn_interval;
                }
            }
            return this.turnIfEnoughAPs(angle_to_turn, false);
        }
    }

    public boolean turnIfEnoughAPs(int angle, boolean turn_to) {
        float cost = 0;
        if (turn_to) {
            int turn_ang = Functions.GetDiffBetweenAngles(angle, this.getAngle());
            cost = (turn_ang * Client.TURN_COST);// / Client.TURN_SPEED;
            if (cost > 4.5) {
                System.out.println("Error: Turning from " + this.getAngle() + " to " + angle +" costs " + cost);
                turn_ang = Functions.GetDiffBetweenAngles(angle, this.getAngle());
            }
            //System.out.println("Turning from " + this.y_angle + " to " + angle +" costs " + cost);
        } else {
            cost = (ssmith.lang.Functions.mod(angle) * Client.TURN_COST) / Client.TURN_SPEED;
        }
        if (this.getAPs() >= cost) {
            super.turnRegardless(angle, turn_to, true, false);
            this.reduceAPs(cost, true);
            return true;
        }
        return false;
    }

    public boolean slideLeftIfEnoughAPs() {
        if (System.currentTimeMillis() > this.action_time) {
            float cost = move_fwd_cost * Client.MOVE_SPEED;
            cost = PlayerObject3D.adjustCostForArmour(this.armour_damage_reduction, cost);
            if (this.getAPs() >= cost) {
                if (attemptSlideLeft()) {
                    this.reduceAPs(cost, true);
                    Server.game.mission.trooperEnteredSquare(this, this.getMapSquare());
                }
                return true;
            }
        }
        return false;
    }

    public boolean slideRightIfEnoughAPs() {
        if (System.currentTimeMillis() > this.action_time) {
            float cost = move_fwd_cost * Client.MOVE_SPEED;
            cost = PlayerObject3D.adjustCostForArmour(this.armour_damage_reduction, cost);
            if (this.getAPs() >= cost) {
                if (attemptSlideRight()) {
                    this.reduceAPs(cost, true);
                    Server.game.mission.trooperEnteredSquare(this, this.getMapSquare());
                }
                return true;
            }
        }
        return false;
    }

    public void damage(int amt, int angle_of_attack) {
        // Adjust damage based on angle of hit
        int armour_reduc = this.armour_damage_reduction;
        if (this.isWithinAngle(angle_of_attack, 23)) {
            // Do nothing. Hit from front.
        } else if (this.isWithinAngle(angle_of_attack, 135)) {
            // Hit from side
            armour_reduc = armour_reduc / 2;
        } else {
            // Hit from back
            armour_reduc = armour_reduc / 4;
        }

        // Adjust the damage by a random value
        amt = (int) Functions.rndDouble(amt / 2, amt * 1.5);
        amt -= armour_reduc;
        //System.out.println(this.name + " lost " + amt + " health.");
        if (amt > 0) {
            //System.out.println("Current armour:"+this.armour_damage_reduction);
            int armour_dam = amt / 5;
            //System.out.println("Armour damage:" + armour_dam+"\n");
            S2CCommunication.SendMessageToAll(this.name + " lost " + Math.min(amt, this.getHealth()) + " health.", (byte)2);
            this.reduceArmour(armour_dam);
            this.reduceHealth(amt);
        }
    }

    /**
     * This function is used for autofire, which only works if they have a weapon.
     */
    public boolean shoot() {
        if (this.current_item != null) {
            if (this.current_item instanceof SWeaponCarried) {
                return this.current_item.use(this);
            }
        }
        return false;
    }

    public void remove(boolean killed) {
        Server.game.removeUnitFromSide(this);
        if (isDestroyed() == false) {
            if (killed) {
                // todo - put these in subclasses instead
                if (this instanceof SPlayerObjectTrooper || this instanceof SPlayerObjectPrisoner || this instanceof SPlayerObjectSternerRegnix) {
                    S2CCommunication.SendScenery(S2CCommunication.TROOPER_CORPSE,
                            this.getX(), this.getZ(), this.getAngle());
                } else if (this instanceof SPlayerObjectRobot) {
                    S2CCommunication.SendScenery(S2CCommunication.ROBOT_CORPSE,
                            this.getX(), this.getZ(), this.getAngle());
                    S2CCommunication.SendScenery(S2CCommunication.SMOKE, this.getX(), this.getZ(), this.getAngle());
                } else {
                    S2CCommunication.SendScenery(S2CCommunication.CORPSE, this.getX(),
                            this.getZ(), this.getAngle());
                }
                S2CCommunication.SendMessageToAll("** " + this.name
                        + " HAS BEEN KILLED! **", (byte) 2);
                Server.game.mission.trooperKilled(this);
            } else {
                S2CCommunication.SendMessageToAll("** " + this.name
                        + " HAS ESCAPED! **", (byte) 2);
            }

            if (this.current_controller != null) {
                this.current_controller.deselectCurrentUnit();
            }

            S2CCommunication.SendVPUpdate(this.side);
            super.remove(killed);
        }
    }

    /** This is PRIVATE as it shouldn't be called from anywhere else!
     *
     * @return
     * @throws IOException
     */
    private CollisionDetails attemptMoveForwards() {
        //System.out.println(this.name + " is moving forwards");
        no_move_reason.reset();
        super.moveForwardsRegardless();
        if (this.canTraverseMap() == false) {
            no_move_reason.setBlockedByScenery();
        } else if (this.checkCollisions(no_move_reason)) {
            //Do nothing;
        }
        if (no_move_reason.moveSuccessful() == false) {
            super.moveBackwardsRegardless(Client.MOVE_SPEED);
        }
        else {
            S2CCommunication.SendObjectUpdateToAll(this, false);
        }
        return no_move_reason;
    }

    /** This is PRIVATE as it shouldn't be called from anywhere else!
     *
     * @return
     * @throws IOException
     */
    private boolean attemptMoveBackwards() {
        super.moveBackwardsRegardless(Client.MOVE_SPEED * 0.6f);
        if (this.canTraverseMap() == false || this.checkCollisions(null)) {
            super.moveForwardsRegardless();
            return false;
        }
        else {
            S2CCommunication.SendObjectUpdateToAll(this, false);
            return true;
        }
    }

    /** This is PRIVATE as it shouldn't be called from anywhere else!
     *
     * @return
     * @throws IOException
     */
    private boolean attemptSlideLeft() {
        super.slideLeftRegardless(Client.MOVE_SPEED * Statics.SLIDE_MOD);
        if (this.canTraverseMap() == false || this.checkCollisions(null)) {
            super.slideRightRegardless(Client.MOVE_SPEED * Statics.SLIDE_MOD);
            return false;
        }
        else {
            S2CCommunication.SendObjectUpdateToAll(this, false);
            return true;
        }
    }

    /** This is PRIVATE as it shouldn't be called from anywhere else!
     *
     * @return
     * @throws IOException
     */
    private boolean attemptSlideRight() {
        super.slideRightRegardless(Client.MOVE_SPEED * Statics.SLIDE_MOD);
        if (this.canTraverseMap() == false || this.checkCollisions(null)) {
            super.slideLeftRegardless(Client.MOVE_SPEED * Statics.SLIDE_MOD);
            return false;
        }
        else {
            S2CCommunication.SendObjectUpdateToAll(this, false);
            return true;
        }
    }

    public SMapSquare getFacingMapSquare() {
        // Get the next map square
        float x = this.getX();
        float z = this.getZ();

        while ((int)x == (int)this.getX() && (int)z == (int)this.getZ()) {
            float off_x = (float) Math.sin(Math.toRadians(getAngle())) * 0.5f;
            x -= off_x;
            float off_z = (float) Math.cos(Math.toRadians(getAngle())) * 0.5f;
            z -= off_z;
        }

        return Server.map.getMapSquare(x, z);
    }

    public SMapSquare getMapSquareAtAbsoluteAngle(int a) {
        // Get the next map square
        float x = this.getX();
        float z = this.getZ();

        while ((int)x == (int)this.getX() && (int)z == (int)this.getZ()) {
            float off_x = (float) Math.sin(Math.toRadians(a)) * 0.5f;
            x = x - off_x;
            float off_z = (float) Math.cos(Math.toRadians(a)) * 0.5f;
            z = z - off_z;
        }

        return Server.map.getMapSquare(x, z);
    }

    public void performActionIfEnoughAPs() {
        if (System.currentTimeMillis() > action_time) {
            SMapSquare sq = this.getFacingMapSquare();
            if (sq.getType() == SMapSquare.OPEN_DOOR) {
                if (this.getAPs() >= OPEN_DOOR_COST) {
                    Server.map.closeDoor(sq);
                    this.reduceAPs(OPEN_DOOR_COST, true);
                    this.action_time = System.currentTimeMillis() + Client.ACTION_DELAY;
                }
            }
            else if (sq.getType() == SMapSquare.CLOSED_DOOR) {
                if (this.getAPs() >= OPEN_DOOR_COST) {
                    Server.map.openDoor(sq);
                    this.reduceAPs(OPEN_DOOR_COST, true);
                    this.action_time = System.currentTimeMillis() + Client.ACTION_DELAY;
                }
            }
            else if (sq.getType() == SMapSquare.ATTACKER_COMPUTER ||
                    sq.getType() == SMapSquare.DEFENDER_COMPUTER) {
                if (this.getAPs() >= TURN_OFF_COMPUTER_COST) {
                    Server.map.turnOffComputer(this, sq);
                    Server.game.mission.computerTurnedOff(sq.getType() == SMapSquare.ATTACKER_COMPUTER ? Client.ATTACKER : Client.DEFENDER, sq);
                    this.reduceAPs(TURN_OFF_COMPUTER_COST, true);
                    this.action_time = System.currentTimeMillis() + Client.ACTION_DELAY;
                }
            }
            else {
                S2CCommunication.SendMessage(this.current_controller, "You cannot do anything here.", (byte)0);
            }
        }
    }

    /*  private int getAmmo() {
     if (this.current_item instanceof SWeaponCarried) {
     SWeaponCarried weapon = (SWeaponCarried) this.current_item;
     return weapon.ammo;
     }
     return 1;
     }
     */
    public void reload(boolean only_if_empty) {
        if (System.currentTimeMillis() > this.action_time) {
            if (this.current_item != null) {
                if (this.current_item instanceof SWeaponCarried) {
                    SWeaponCarried weapon = (SWeaponCarried)this.current_item;
                    if (only_if_empty == false || weapon.ammo <= 0) {
                        weapon.reload(this);
                        this.action_time = System.currentTimeMillis() + Client.RELOAD_DELAY;
                        S2CCommunication.SendAmmoLevel(this, weapon);
                    }
                }
            }
        }
    }

    /*  public void throwGrenade(int timer, int dist) throws IOException {
     if (this.current_item != null) {
     if (this.current_item instanceof SGrenadeCarried) {
     SGrenadeCarried grenade = (SGrenadeCarried)this.current_item;
     grenade.prime(timer);
     grenade.throwItem(this, dist);
     }
     }
     }
     */
    public float getAPs() {
        return this.action_points;
    }

    public void reduceAPs(float amt, boolean send_comms) {
        if (amt == 0) {
            return;
        }
        int old_aps = (int) this.action_points;
        this.action_points = this.action_points - amt;
        //System.out.println("Reduce APs - name " + name + " (" + amt + ")");
        if (this.getAPs() > Server.MAX_ACTION_POINTS) {
            this.action_points = Server.MAX_ACTION_POINTS;
        }
        if (this.guarding && this.getAPs() >= Server.MAX_ACTION_POINTS) {
            this.action_points = Server.MAX_ACTION_POINTS - 1;
        }
        if (send_comms) {
            if (old_aps != (int)this.action_points) {
                S2CCommunication.SendAPs(this);
            }
        }
    }

    public void setAPs(int amt) {
        double old_aps = this.action_points;
        this.action_points = amt;
        if (this.action_points != old_aps) {
            //System.out.println("Set APs - name " + name + " (" + amt + ")");
            S2CCommunication.SendAPs(this);
        }
    }

    public boolean getAutofireSetting() {
        return this.autofire;
    }

    public void setAutofire(boolean auto, boolean send_to_client) {
        if (this.autofire != auto) {
            this.autofire = auto;
            if (send_to_client) {
                S2CCommunication.SendAutofire(this);
            }
        }
    }

    public void setShotType(int type) {
        this.shot_type = type;
    }

    public int getCombat() {
        return this.combat;
    }

    public float getAccuracy() {
        return this.accuracy;
    }

    public boolean isGuarding() {
        return this.guarding;
    }

    public void ensureNoHoldup() {
        if (getAPs() >= Server.MAX_ACTION_POINTS) {
            reduceAPs(0.1f, false); // Stop us holding up the game.
        }

    }

    public void setGuard(boolean sk, boolean send_to_client) {
//      if (sk != this.skip) {
        this.guarding = sk;
//      if (send_to_client) {
        S2CCommunication.SendGuard(this);
//      }
//      }
    }

    public boolean isInMiddleOfSquare_OLD() {
        int x = (int)getX();
        int z = (int)getZ();
        float x2 = getX() - x;
        float z2 = getZ() - z;

        if (x2 > 0.4f && x2 < 0.6f && z2 > 0.4f && z2 < 0.6f) {
            return true;
        }
        return false;
    }

    public boolean isInMiddleOfSquare() {
        SMapSquare sq1 = Server.map.getMapSquare(this.getX() - this.radius, this.getZ() - this.radius);
        SMapSquare sq2 = Server.map.getMapSquare(this.getX() + this.radius, this.getZ() - this.radius);
        if (sq1 == sq2) {
            SMapSquare sq3 = Server.map.getMapSquare(this.getX() - this.radius, this.getZ() + this.radius);
            if (sq1 == sq3) {
                SMapSquare sq4 = Server.map.getMapSquare(this.getX() + this.radius, this.getZ() + this.radius);
                if (sq1 == sq4) {
                    return true;
                }
            }
        }
        return false;
    }

    private static void loadNames() {
        TextFile file = new TextFile();
        firstnames = new ArrayList();
        surnames = new ArrayList();

        try {
            file.openFile(Client.DATA_DIR + "firstnames.csv", TextFile.READ);
            while (file.isEOF() == false) {
                firstnames.add(file.readLine());
            }
            file.close();
        } catch (Exception e) {
            //System.err.println("Error - Cannot find first names file!");
            //e.printStackTrace();
            Server.HandleError(e);
        }


        try {
            file.openFile(Client.DATA_DIR + "surnames.csv", TextFile.READ);
            while (file.isEOF() == false) {
                surnames.add(file.readLine());
            }
            file.close();
        } catch (IOException e) {
            Server.HandleError(e);
        }
    }

    protected static String GetRandomName() {
        try {
            int fname = Functions.rnd(0, firstnames.size()-1);
            int sname = Functions.rnd(0, surnames.size()-1);
            String name = (String)firstnames.get(fname) + " " + (String)surnames.get(sname);
            //firstnames.remove(fname);
            //surnames.remove(sname);
            return name;
        } catch (Exception e) {
        	e.printStackTrace();
            return "Mr. X";
        }
    }

    public String toString() {
        return this.name;
    }

    public void calcRoute(int x, int z, int dist) {
        this.ai.calcRoute(x, z, dist);
    }

    public void useCurrentItem() {
        if (System.currentTimeMillis() > action_time) {
            if (this.current_item.use(this)) {
                action_time = System.currentTimeMillis() + Client.SHOOT_DELAY;
            }
        }
    }

    public CollisionDetails getNoMoveReason() {
        return this.no_move_reason;
    }

    public int getAngleForShootingAccuracy(SWeaponCarried weapon) {
        float acc = this.accuracy + weapon.accuracy;
        // Adjust for shot type
        if (shot_type == SPlayerObject3D.SNAP_SHOT) {
            acc = acc * 2;
        } else if (shot_type == SPlayerObject3D.AIMED_SHOT) {
            acc = acc / 2;
        }
        //turnRegardless((int)Functions.rndFloat( -acc, acc), false, true, true);
        return (int)Functions.rndFloat( -acc, acc);
    }

    public int getShotAPCost(SWeaponCarried weapon) {
        int aps_reqd = weapon.shot_aps;
        // Adjust for shot type
        if (shot_type == SPlayerObject3D.SNAP_SHOT) {
            aps_reqd = aps_reqd / 2;
        } else if (shot_type == SPlayerObject3D.AIMED_SHOT) {
            aps_reqd = aps_reqd * 2;
        }
        if (aps_reqd >= Server.MAX_ACTION_POINTS) {
            aps_reqd = Server.MAX_ACTION_POINTS-1; // In case we're skipping
        }
        return aps_reqd;
    }

    public SPlayerData getCurrentController() {
        return this.current_controller;
    }

    public void setCurrentController(SPlayerData pdata) {
        this.current_controller = pdata;
    }

}
