package ngrave.server.objects;

import ssmith.lang.*;
import ssmith.game.*;
import ssmith.awt.PointF;
import java.io.*;
import java.util.ArrayList;

import ngrave.client.Client;
import ngrave.server.CollisionDetails;
import ngrave.server.S2CCommunication;
import ngrave.server.SMapSquare;
import ngrave.server.Server;
import ngrave.server.ServerFunctions;

public abstract class SObject implements ICollidableObject {

	public static final int VIEW_ANGLE = 55;

	private float x_pos, z_pos, speed;
	public final float radius; // Used for collision detection.
	private int y_angle;
	private int id; // This is the same as it's array cell;
	private boolean destroyed = false;
	private final boolean can_collide;
	public final byte side;
	private PointF pf_lhs = new PointF(0f, 0f);
	private PointF pf_rhs = new PointF(0f, 0f);
	private PointF move_dir = new PointF(0f, 0f);

	public SObject(float x, float z, int ang, float diameter, byte side, boolean scenery, float spd) {
		this.x_pos = x;
		this.z_pos = z;
		this.y_angle = ang;
		this.radius = diameter/2;
		this.side = side;
		this.can_collide = !scenery;
		this.speed = spd;
		this.calcMoveDir(); // Must be after speed is set!

		boolean found = false;
		while (!found) {
			for (int o = 0; o < Server.objects.length; o++) {
				if (Server.objects[o] == null) {
					Server.objects[o] = this;
					this.id = o;
					found = true;
					break;
				}
			}
			if (!found) {
				// Rebuild array
				SObject new_arr[] = new SObject[Server.objects.length * 2];
				System.arraycopy(Server.objects, 0, new_arr, 0, Server.objects.length);
				Server.objects = new_arr;
				System.out.println("Rebuilt Server.objects[].");
			}
		}

		if (this.can_collide) {
			Server.getCollisionMatrix().addObject(this);
		}

		if (Client.SHOW_OBJ_CREATION) {
			System.out.println("object " + this.toString() + "(" + id + ") created.");
		}
	}

	public String toString() {
		return this.getClass().toString();
	}

	public abstract void process();

	public int getID() {
		return this.id;
	}

	public float getSpeed() {
		return this.speed;
	}

	public void setSpeed(float s) {
		this.speed = s;
		this.calcMoveDir();
	}

	public int getAngle() {
		return this.y_angle;
	}

	public float getX() {
		return this.x_pos;
	}

	public float getZ() {
		return this.z_pos;
	}

	public float getRadius() {
		return this.radius;
	}

	public boolean isDestroyed() {
		return this.destroyed;
	}

	public void setDestroyed() {
		this.destroyed = true;
	}

	protected boolean checkCollisions(CollisionDetails nmr) {
		boolean result = false;
		ArrayList objs = Server.getCollisionMatrix().getPotentialColliders(this);
		SObject obj;
		int size = objs.size();
		for (int o=0 ; o<size ; o++) {
			//System.out.println("Type:" + objs.get(o).toString());
			obj = (SObject)objs.get(o);
			// See if any objects have collided.
			if (this.isWithinCollidingDistance(obj)) {
				if (this.hasCollidedWith(obj)) {
					if (nmr != null) {
						nmr.setBlockedByObject(obj);
					}
					result = true;
				}
				if (this.destroyed) {
					// No point continuing.
					return result;
				}
			}
		}
		return result;
	}

	public boolean canSee(SObject target, boolean check_direction) {
		if (target.destroyed) {
			return false;
		}
		return canSee(target.x_pos, target.z_pos, check_direction);
	}

	public boolean canSee(SMapSquare target, boolean check_direction) {
		return canSee(target.x + 0.5f, target.z + 0.5f, check_direction);
	}

	/**
	 * There is also an equivalent function in CObject for
	 * the client to work it out itself.
	 */
	public boolean canSee(float tx, float tz, boolean check_direction) {
		if (Functions.distance(this.x_pos, this.z_pos, tx, tz) >
		Client.MAX_VIEW_DIST) {
			return false;
		}
		if (check_direction) {
			int ang_to_target = this.getAbsoluteAngleTo(tx, tz);
			if (this.isWithinAngle(ang_to_target, VIEW_ANGLE) == false) {
				return false;
			}
		}

		//PointF pf = this.getLeftSidePoint(tx, tz);
		if (ServerFunctions.CanSee(pf_lhs.x, pf_lhs.y, tx, tz)) { // Check we can see it from out LHS
			//pf =  getRightSidePoint(tx, tz);
			return ServerFunctions.CanSee(pf_rhs.x, pf_rhs.y, tx, tz); // Check we can see it from our RHS
		}
		return false;
	}

	public void setLeftSidePoint() {
		float off_x = (float) Math.sin(Math.toRadians(this.y_angle+90)) * this.radius;
		pf_lhs.x = this.x_pos - off_x;
		float off_z = (float) Math.cos(Math.toRadians(this.y_angle+90)) * this.radius;
		pf_lhs.y = this.z_pos - off_z;
	}

	public void setRightSidePoint() {
		float off_x = (float) Math.sin(Math.toRadians(this.y_angle-90)) * this.radius;
		pf_rhs.x = this.x_pos - off_x;
		float off_z = (float) Math.cos(Math.toRadians(this.y_angle-90)) * this.radius;
		pf_rhs.y = this.z_pos - off_z;
	}

	/*
  public PointF getLeftSidePoint(float tx, float tz) {
      int ang_to = this.getAbsoluteAngleTo(tx, tz);
      float off_x = (float) Math.sin(Math.toRadians(ang_to+90)) * this.radius;
      tmp_pf.x = this.x_pos - off_x;
      float off_z = (float) Math.cos(Math.toRadians(ang_to+90)) * this.radius;
      tmp_pf.y = this.z_pos - off_z;
      return tmp_pf;
  }

  public PointF getRightSidePoint(float tx, float tz) {
      int ang_to = this.getAbsoluteAngleTo(tx, tz);
      float off_x = (float) Math.sin(Math.toRadians(ang_to-90)) * this.radius;
      tmp_pf.x = this.x_pos - off_x;
      float off_z = (float) Math.cos(Math.toRadians(ang_to-90)) * this.radius;
      tmp_pf.y = this.z_pos - off_z;
      return tmp_pf;
  }
	 */
	protected boolean isWithinCollidingDistance(SObject other) {
		if (Functions.distance(this.x_pos, this.z_pos, other.x_pos, other.z_pos) <= (this.radius+other.radius)) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Over-ride this to decide what happens when two objects collide.
	 * Return whether movement is blocked.
	 */
	public boolean hasCollidedWith(SObject other) {
		return false;
	}

	public void turnRegardless(int angle, boolean turn_to, boolean send_comms, boolean force) {
		//int old_angle = this.y_angle;
		if (turn_to) {
			if (angle == this.y_angle) {
				return;
			}
			this.y_angle = angle;
		} else {
			if (angle == 0) {
				return;
			}
			this.y_angle += angle; // Left is +, Right is -
			if (this.y_angle < 0) {
				this.y_angle += 360;
			}
			if (this.y_angle > 360) {
				this.y_angle -= 360;
			}
		}

		this.calcMoveDir();

		if (send_comms) {
			//if (Functions.mod(old_angle - this.y_angle) >= Client.TURN_SPEED) {
			S2CCommunication.SendObjectTurnUpdateToAll(this, force);
			//}
		}
	}

	private void calcMoveDir() {
		this.move_dir.x = (float) Math.sin(Math.toRadians(y_angle)) * speed;
		this.move_dir.y = (float) Math.cos(Math.toRadians(y_angle)) * speed;
		this.setLeftSidePoint();
		this.setRightSidePoint();
	}

	public void remove(boolean killed) {
		if (this.can_collide) {
			Server.getCollisionMatrix().removeObject(this);
		}
		S2CCommunication.SendRemoveObject(this, killed);
		this.destroyed = true;
		Server.objects[this.id] = null;
		if (Client.SHOW_OBJ_CREATION) {
			System.out.println("Object " + this.toString() + " removed.");
		}
	}

	protected void setLocation(float x, float z, int angle, boolean force) {
		if (this.can_collide) {
			Server.getCollisionMatrix().removeObject(this);
		}
		this.x_pos = x;
		this.z_pos = z;
		if (this.can_collide) {
			Server.getCollisionMatrix().addObject(this);
		}
		this.turnRegardless(angle, true, false, true);
		S2CCommunication.SendObjectUpdateToAll(this, force);
	}

	public void setLocation(float x, float z, boolean force) {
		this.setLocation(x, z, this.y_angle, force);
	}

	/**
	 * @param dist
	 * @return
	 * @throws IOException
	 */
	public void moveForwardsRegardless() {
		if (this.can_collide) {
			Server.getCollisionMatrix().removeObject(this);
		}
		if (move_dir.x == 0 && move_dir.y == 0) {
			System.err.println("Unit " + this.toString() + " has no move dir!");
		}
		//float x = (float) Math.sin(Math.toRadians(y_angle)) * dist;
		x_pos -= this.move_dir.x;
		//float z = (float) Math.cos(Math.toRadians(y_angle)) * dist;
		z_pos -= this.move_dir.y;
		if (this.can_collide) {
			Server.getCollisionMatrix().addObject(this);
		}
	}

	public void moveBackwardsRegardless(float dist) {
		if (this.can_collide) {
			Server.getCollisionMatrix().removeObject(this);
		}
		float x2 = (float) Math.sin(Math.toRadians(y_angle)) * dist;
		x_pos += x2;
		float z2 = (float) Math.cos(Math.toRadians(y_angle)) * dist;
		z_pos += z2;
		if (this.can_collide) {
			Server.getCollisionMatrix().addObject(this);
		}
	}

	protected boolean slideLeftRegardless(float dist) {
		if (this.can_collide) {
			Server.getCollisionMatrix().removeObject(this);
		}
		float x = (float) Math.sin(Math.toRadians(y_angle+90)) * dist;
		x_pos -= x;
		float z = (float) Math.cos(Math.toRadians(y_angle+90)) * dist;
		z_pos -= z;
		if (this.can_collide) {
			Server.getCollisionMatrix().addObject(this);
		}
		return true;
	}

	protected boolean slideRightRegardless(float dist) {
		if (this.can_collide) {
			Server.getCollisionMatrix().removeObject(this);
		}
		float x = (float) Math.sin(Math.toRadians(y_angle-90)) * dist;
		x_pos -= x;
		float z = (float) Math.cos(Math.toRadians(y_angle-90)) * dist;
		z_pos -= z;
		if (this.can_collide) {
			Server.getCollisionMatrix().addObject(this);
		}
		return true;
	}

	public boolean canTraverseMap() {
		return canTraverseMapTL() && canTraverseMapTR() && canTraverseMapBL() && canTraverseMapBR();
	}

	private boolean canTraverseMapTL() {
		return canTraverse(Server.map.getMapSquare(this.x_pos+(radius), this.z_pos-(radius)));
	}

	private boolean canTraverseMapTR() {
		return canTraverse(Server.map.getMapSquare(this.x_pos+(radius), this.z_pos+(radius)));
	}

	private boolean canTraverseMapBL() {
		return canTraverse(Server.map.getMapSquare(this.x_pos-(radius), this.z_pos-(radius)));
	}

	private boolean canTraverseMapBR() {
		return canTraverse(Server.map.getMapSquare(this.x_pos-(radius), this.z_pos+(radius)));
	}

	public abstract boolean canTraverse(SMapSquare sq);

	public float distanceTo(SObject object) {
		return (float) Functions.distance(this.x_pos, this.z_pos, object.x_pos, object.z_pos);
	}

	public float distanceTo(SMapSquare sq) {
		return (float) Functions.distance(this.x_pos, this.z_pos, sq.x + 0.5f, sq.z + 0.5f);
	}

	/**
	 *
	 */
	 public int getAbsoluteAngleTo(SObject obj) {
		 return getAbsoluteAngleTo(obj.x_pos, obj.z_pos);
	 }

	 public int getAbsoluteAngleTo(float x, float z) {
		 x = this.x_pos - x;
		 z = this.z_pos - z;
		 if (x>=0 && z>=0) {
			 return (int) Math.toDegrees(Math.atan(x/z));
		 } else if (x<0 && z>=0) {
			 return (int) Math.toDegrees(Math.atan(x/z)) + 360;
		 } else {
			 return (int) Math.toDegrees(Math.atan(x/z)) + 180;
		 }
	 }

	 /**
	  *
	  */
	 public int getAbsoluteAngleTo(SMapSquare sq) {
		 float x = this.x_pos - (sq.x + 0.5f);
		 float z = this.z_pos - (sq.z + 0.5f);
		 if (x>=0 && z>=0) {
			 return (int) Math.toDegrees(Math.atan(x/z));
		 } else if (x<0 && z>=0) {
			 return (int) Math.toDegrees(Math.atan(x/z)) + 360;
		 } else {
			 return (int) Math.toDegrees(Math.atan(x/z)) + 180;
		 }
	 }

	 /**
	  * Returns whether one angle is within so many degrees of another.
	  */
	 public boolean isWithinAngle(int deg, int within) {
		 float diff = Functions.GetDiffBetweenAngles(deg, this.y_angle);
		 return diff <= within;
	 }

}
