/*
 * Created on 05-Oct-2006
 *
 */
package ngrave.server;

import java.io.IOException;
import java.util.ListIterator;

import ngrave.client.Client;
import ngrave.missions.Mission;
import ngrave.server.objects.SObject;
import ngrave.server.objects.SPlayerObject3D;
import ssmith.game.CollisionMatrix2D;

public final class SGame {

    private static final long RESTART_DELAY = 10000;

    // These cannot be changed
    public Mission mission; // todo - make private
    private final boolean random_map_selected;

    private byte winner;
    //private byte default_cpu_side;
    private boolean b_time_expired;
    private float elapsed_time;
    private SSideData[] side_data;
    private CollisionMatrix2D coll_matrix;
    private long restart_at_time;
    private boolean attackers_are_cpu, defenders_are_cpu;
    
    public SGame(String mission_name, boolean random_map, boolean attack_are_cpu, boolean defend_are_cpu) throws ClassNotFoundException, InstantiationException, IllegalAccessException {
        side_data = new SSideData[2];
        side_data[Client.ATTACKER] = new SSideData(Client.ATTACKER);
        side_data[Client.DEFENDER] = new SSideData(Client.DEFENDER);

        this.random_map_selected = random_map;
        Class mission_class = Class.forName("ngrave.missions." + mission_name);
        mission = (Mission)mission_class.newInstance();
        //default_cpu_side = def_cpu_side;
        attackers_are_cpu = attack_are_cpu;
        defenders_are_cpu = defend_are_cpu;
    }

    public CollisionMatrix2D getCollisionMatrix() {
        if (coll_matrix == null) {
            coll_matrix = new CollisionMatrix2D(Server.map.size+1, false);
        }
        return coll_matrix;
      }

    public void setupNewGame() {
        coll_matrix = null;
        winner = -1;
        b_time_expired = false;
        elapsed_time = 0;
        restart_at_time = 0;
        Server.objects = new SObject[Server.START_OBJECTS];
        side_data[Client.ATTACKER].reset();
        side_data[Client.DEFENDER].reset();

        System.gc();

        ListIterator players = Server.players.getIterator();
        while (players.hasNext()) {
            SPlayerData player = (SPlayerData) players.next();
            player.deselectCurrentUnit(); // In case we have one
            S2CCommunication.SendGameData(player);
            S2CCommunication.SendNewMap(player);
        }

        mission.setupForServer();
        if (attackers_are_cpu) {
            mission.setupAI(Client.ATTACKER);
            this.side_data[Client.ATTACKER].controlled_by_cpu = true; // todo -make part of setupAI
        }
        if (defenders_are_cpu) {
            mission.setupAI(Client.DEFENDER);
            this.side_data[Client.DEFENDER].controlled_by_cpu = true; // todo -make part of setupAI
        }
        
        S2CCommunication.SendElapsedTime();

        side_data[Client.ATTACKER].logIPs();
        side_data[Client.DEFENDER].logIPs();
    }

    public boolean isRandomMapSelected() {
        return this.random_map_selected;
    }

    /**
     * This is what gets called to end the game
     * @param side
     * @throws IOException
     */
    public void sideHasWon(byte side) {
    	if (winner < 0) {
        S2CCommunication.SendWinner(side);
        winner = side;
        restart_at_time = System.currentTimeMillis() + RESTART_DELAY;
        S2CCommunication.SendMessageToAll("Starting new game iminently.  Disconnect now if you want to change side!", (byte)1);
    	}
    }

    public float getElapsedTime() {
        return elapsed_time;
    }

    public void incVPs(byte side, int amt) {
        side_data[side].incVPs(amt);
    }

    public byte getVPs(byte side) {
        return side_data[side].getVPs();
    }

    public void incElapsedTime(float amt) throws IOException {
        this.elapsed_time += amt;

        S2CCommunication.SendElapsedTime();
        if (b_time_expired == false) {
        if (mission.getMaxTime() > 0) {
            if (getElapsedTime() > mission.getMaxTime()) {
                    b_time_expired = true;
                    S2CCommunication.SendMessageToAll("** TIME UP**", (byte)2);
                    mission.timeExpired();
                }
            }
        }
    }

    public void addPlayerToSide(byte side) {
        side_data[side].addPlayer();
    }

    public void removePlayerFromSide(byte side) {
        if (side >= 0) {
            side_data[side].removePlayer();
        }
    }

    public byte getNoOfPlayersOnSide(byte side) {
        return side_data[side].getNoOfPlayers();
    }

    public byte getMaxNoOfPlayersOnSide(byte side) {
        return side_data[side].getMaxNoOfPlayers();
    }

    public byte getTotalNoOfPlayers() {
        return (byte)(side_data[0].getNoOfPlayers() + side_data[1].getNoOfPlayers());
    }

    public byte getMaxNoOfPlayers() {
        return (byte)(side_data[0].getMaxNoOfPlayers() + side_data[1].getMaxNoOfPlayers());
    }

    public void addUnitToSide(SPlayerObject3D obj) {
        side_data[obj.side].addUnit(obj);
    }

    public void removeUnitFromSide(SPlayerObject3D obj) {
        side_data[obj.side].removeUnit(obj);
    }

    public byte getNoOfUnitsOnSide(byte side) {
        return side_data[side].getNoOfUnits();
    }

    public byte getTotalNoOfUnits() {
        return (byte)(side_data[0].getNoOfUnits() + side_data[1].getNoOfUnits());
    }

    public void unitIsNowCPU(SPlayerObject3D obj) {
        side_data[obj.side].unitIsNowCPU(obj);
    }

    public void process() {
        if (this.winner >= 0) {
            if (System.currentTimeMillis() > restart_at_time) {
                this.setupNewGame();
            }
        }
    }

    public boolean isControlledByCPU(byte side) {
        return this.side_data[side].controlled_by_cpu;
    }

    public void setControlledByCPU(byte side, boolean b) {
        this.side_data[side].controlled_by_cpu = b;
    }

}


