package ngrave.client.objects;

import ssmith.lang.*;
import java.io.*;

import ssmith.opengl.Canvas3D;
import ssmith.opengl.I3DObject;
import net.java.games.jogl.*;
import ngrave.client.Client;
import ngrave.client.IClientControlled;
import ngrave.server.objects.SObject;

/**
 * This is the base class for all drawable/processable objects of the client.
 * This class will add the item to the collection automatically.
 * @author stephen smith
 *
 */
public abstract class CObject implements I3DObject {

  protected float x_pos, z_pos;
  public float y_pos;
  public int y_angle=0;
  private boolean destroyed = false;
  private final boolean draw_on_map;
  private boolean can_be_seen = false; // False by default
  public final int id;
  private final byte owner;

  public CObject(int n, float x, float y, float z, int y_ang, byte owner) {
  	this.id = n;
  	this.x_pos = x;
  	this.y_pos = y;
  	this.z_pos = z;
  	this.y_angle = y_ang;
  	this.owner = owner;
  	this.draw_on_map = (owner == Client.players.getThisPlayer().side || n < 0); // Draw on map if scenery or owned by us.

  	if (id >= 0) { // As non-server controlled objects are given a no of -1.
  		// Check the array is big enough.
  		if (id >= Client.objects.length) {
  			// Rebuild array
  			CObject new_arr[] = new CObject[Client.objects.length * 2];
  			System.arraycopy(Client.objects, 0, new_arr, 0, Client.objects.length);
  			Client.objects = new_arr;
  			System.out.println("Rebuilt Client.objects[]: " + Client.objects.length);
  		}
  		Client.objects[id] = this;
  	} else {
  		// Add it to scenery.
  		if (this instanceof IClientControlled) {
  			Client.client_processed_objects.add(this);
  		} else {
  			System.err.println("Warning: " + this.toString() + " not client controlled.");
  		}
  	}

  	this.checkIfCanBeSeen();

  	Client.canvas.addObject(this);
  }

  public void remove() {
  	this.destroyed = true;

  	Client.canvas.removeObject(this);
  	if (this.id >= 0) {
  		Client.objects[this.id] = null;
  		if (Client.units_on_my_side.contains(this)) {
  			Client.units_on_my_side.remove(this);
  			if (this == Client.camera.getViewer()) {
  				Client.camera.showTopDown();
  			}
  			if (this == Client.current_unit) {
  				Client.current_unit = null;
  			}
  		}
  	} else {
  		Client.client_processed_objects.remove(this);
  	}
  }

  public boolean isDestroyed() {
      return this.destroyed;
  }

  public byte getOwner() {
  	return owner;
  }

  /**
   * There is also an equivalent function in SObject3D.
   */
  public boolean canSee(CObject target) {
  	if (target.destroyed) {
  		return false;
  	}
  	if (Functions.distance(this.x_pos, this.z_pos, target.x_pos, target.z_pos) > Client.MAX_VIEW_DIST) {
  		return false;
  	}

  	int ang_to_target = this.getAngleTo(target);
  	if (this.isWithinAngle(ang_to_target, SObject.VIEW_ANGLE)) {
  		float sx = this.x_pos;
  		float sz = this.z_pos;
  		float x = sx;
  		float z = sz;
  		float tx = target.x_pos;
  		float tz = target.z_pos;

  		double iterations = Functions.distance(sx, sz, tx, tz) * 4;
  		double x_off = (tx - x) / iterations;
  		double z_off = (tz - z) / iterations;
  		int last_x = -1;
  		int last_z = -1;
        iterations = iterations - 2; // So we don't go too far
  		for (int s = 0; s < iterations; s++) {
  			x += x_off;
  			z += z_off;
  			if (last_x != (int) x || last_z != (int) z) {
  				if (Client.map.getMapSquare((int)x, (int)z).blocks_targetting_view) {
  					return false;
  				}
  			}
  			last_x = (int)x;
  			last_z = (int)z;
  		}
  		return true;
  	}
  	return false;
  }

  public boolean shouldBeDrawn() {
      if (Client.camera.isLockedToTrooper()) {
          // Viewing from units eyes
          if (Client.camera.getViewer() == this) {
              // Don't draw ourselves
              return false;
          } else {
              // Draw them if we can see them.
              return Client.camera.getViewer().canSee(this);
          }
      } else {
          // Map mode!
          if (Client.camera.getViewer() != null) {
              // We have selected a unit to control
              return this.can_be_seen || this.draw_on_map || Client.camera.getViewer().canSee(this);
          } else {
              // FSR, no current unit selected
              return this.draw_on_map;
          }
      }
  }

	public void checkIfCanBeSeen() {
		if (draw_on_map == false) { // Only objects that aren't drawn anyway
                    CObject trooper;
                    // Loop through all our units to see if they can see this object
                    if (Client.units_on_my_side != null) {
                        int size = Client.units_on_my_side.size();
                        for (int s = 0; s < size; s++) {
                            trooper = (CObject) Client.units_on_my_side.get(s);
                            if (trooper.canSee(this)) {
                                this.can_be_seen = true;
                                return;
                            }
                        }
                    }
                    this.can_be_seen = false;
                }
	}

  public abstract void draw(GL gl, GLU glu) throws IOException;

  public void drawCrossHairs(GL gl, GLU glu) {
  	this.moveToPosition(gl);
  	// Draw crosshairs
  	gl.glColor4f(1f, 1f, 1f, 1f);
  	gl.glDisable(GL.GL_TEXTURE);
  	gl.glBegin(GL.GL_LINES);
  	// Horiz
  	final float len = 0.1f;
  	gl.glVertex3f(len, 0f, -1f);
  	gl.glVertex3f(-len, 0f, -1f);
  	// Vert
  	gl.glVertex3f(0f, len, -1f);
  	gl.glVertex3f(0f, -len, -1f);
  	gl.glEnd();
  	if (Canvas3D.texture_on) {
  		gl.glEnable(GL.GL_TEXTURE);
  	}
  }

  /**
   * This is used to position us ready for drawing.
   */
  public void moveToPosition(GL gl) {
  	// Move to the objects position
  	gl.glTranslatef(x_pos, y_pos, z_pos);
  	gl.glRotatef((float)y_angle, 0.0f, 1.0f, 0.0f); // Spin to objects angle;
  }

  /**
   * This is used to position us ready for drawing.
   */
  public void returnToOrigin(GL gl) {
  	// Move to the objects position
  	gl.glRotatef((float)-y_angle, 0.0f, 1.0f, 0.0f); // Spin to objects angle;
  	gl.glTranslatef(-x_pos, -y_pos, -z_pos);
  }

  public boolean canSee(CMapSquare sq, boolean only_check_angle) {
	  if (only_check_angle == false) {
          // Check distance
		  double dist = this.distanceTo(sq);
		  if (dist > Client.MAX_VIEW_DIST) {
			  return false;
		  }
	  }
      // Check angle
	  int ang_to_target = this.getAngleTo(sq);
	  if (this.isWithinAngle(ang_to_target, SObject.VIEW_ANGLE)) {
		  if (only_check_angle == false) {
              // Check LOS
			  if (this.canSeeOff(sq, 0, 0) == false) {
				  if (this.canSeeOff(sq, 0.99f, 0) == false) {
					  if (this.canSeeOff(sq, 0, 0.99f) == false) {
						  if (this.canSeeOff(sq, 0.99f, 0.99f) == false) {
							  return false;
						  }
					  }
				  }
			  }
		  }
		  return true;
	  } else {
		  return false;
	  }
  }

  /**
   * Returns whether one angle is within so many degrees of another.
   */
  public boolean isWithinAngle(int deg, int within) {
  	float diff = deg - this.y_angle;
  	if (diff >= 180) {
  		diff -= 360;
  	} else if (diff < -180) {
  		diff += 360;
  	}
  	return (Functions.mod(diff) <= within);
  }

  public double distanceTo(CMapSquare sq) {
  	return Functions.distance(this.x_pos, this.z_pos, sq.x, sq.z);
  }

  public boolean canSeeOff(CMapSquare sq, float off_x, float off_z) {
  	float sx = this.x_pos;
  	float sz = this.z_pos;
  	float x = sx;
  	float z = sz;
  	float tx = sq.x + off_x;
  	float tz = sq.z + off_z;
  	double iterations = Functions.distance(sx, sz, tx, tz) * 2;
  	double x_off = (tx - x) / iterations;
  	double z_off = (tz - z) / iterations;
  	int last_x = -1;
  	int last_z = -1;
  	for (int s = 1; s < iterations; s++) {
  		x += x_off;
  		z += z_off;
  		if (last_x != (int) x || last_z != (int) z) {
  			if ((int)x == (int)tx && (int)z == (int)tz) {
  				return true;
  			}
  			if (Client.map.getMapSquare((int)x, (int)z).blocks_scenery_view) {
  				return false;
  			}
  		}
  		last_x = (int)x;
  		last_z = (int)z;
  	}
  	return true;
  }

  public int getAngleTo(CObject obj) {
  	float x = this.x_pos - obj.x_pos;
  	float z = this.z_pos - obj.z_pos;
  	if (x>=0 && z>=0) {
  		return (int) Math.toDegrees(Math.atan(x/z));
  	} else if (x<0 && z>=0) {
  		return (int) Math.toDegrees(Math.atan(x/z)) + 360;
  		/*    } else if (x>0 && z<0) {
  		 return (int) Math.toDegrees(Math.atan(x/z)) + 180;*/
  	} else {
  		return (int) Math.toDegrees(Math.atan(x/z)) + 180;
  	}
  }

  public void animateMovement() {
  	// To be overridden
  }

  /**
   *
   */
  public int getAngleTo(CMapSquare sq) {
  	float x = this.x_pos - (sq.x + 0.5f);
  	float z = this.z_pos - (sq.z + 0.5f);
  	if (x>=0 && z>=0) {
  		return (int) Math.toDegrees(Math.atan(x/z));
  	} else if (x<0 && z>=0) {
  		return (int) Math.toDegrees(Math.atan(x/z)) + 360;
  		/*    } else if (x>0 && z<0) {
  		 return (int) Math.toDegrees(Math.atan(x/z)) + 180;*/
  	} else {
  		return (int) Math.toDegrees(Math.atan(x/z)) + 180;
  	}
  }

  public void setLocation(float x, float z) {
    this.x_pos = x;
    this.z_pos = z;
}


}
