package ngrave.client;

import java.io.DataOutputStream;
import java.io.IOException;
import java.io.DataInputStream;

import ngrave.client.objects.CObject;
import ngrave.client.objects.CPlayerObject3D;
import ngrave.server.ClientConnection;
import ngrave.server.S2CCommunication;
import ngrave.server.SPlayerData;
import ngrave.server.Server;
import ngrave.server.ServerFunctions;
import ngrave.server.items.SGrenadeCarried;
import ngrave.server.objects.SObject;
import ngrave.server.objects.SPlayerObject3D;
import ngrave.shared.PlayerObject3D;
import ssmith.lang.Functions;

/**
 * This class contains all the functions for sending data to the server, along with the 
 * corresponding functions to decode it.
 * @author stevec
 *
 */
public final class C2SCommunication {

	private static final int MAX_STRING_LENGTH = 1024;

	// Commands
	public static final byte AUTOFIRE=1, GUARD=2, RELOAD=3, ACTION=4, USE_ITEM=5;
	public static final byte CHANGED_ITEM=6, ANGLE=7;
	public static final byte PLAYERS_DATA=13, PRIME_AND_THROW=14;
	public static final byte KEY_PRESSED=16, KEY_RELEASED=17;
	public static final byte CHANGED_TROOPER=18, WAIT=19, SHOT_TYPE=20;
	public static final byte BUY_UNIT=21, DISCONNECT=22, PLAYERS_NAME=23, MSG=24;
	public static final byte PING=25, SELECT_NEXT_UNIT=26, SELECT_PREV_UNIT=27;
	public static final byte PLAY_AGAINST_CPU=28, SIDE_SELECTED=29;
	public static final byte DESELECT_UNIT=30, LOCATION=31;
	public static final byte OBJECTS_COLLIDED=32, VOTE_SENT=33;
	public static final byte CHECK=99;

	private C2SCommunication() {
	}

	public static void WriteString(DataOutputStream dos, String str) throws IOException {
		byte[] b = str.getBytes();
		dos.writeInt(b.length);
		dos.write(b);
	}

	public static String ReadString(DataInputStream dis) throws IOException {
		int len = dis.readInt();
		if (len <= MAX_STRING_LENGTH) {
			byte b[] = new byte[len];
			long wait_until = System.currentTimeMillis() + 1000;
			while (dis.available() < len) {
				Functions.delay(100);
				if (System.currentTimeMillis() > wait_until) {
					throw new IOException("Error waiting for string.");
				}
			}
			dis.read(b);
			return new String(b);
		} else {
			//throw new RuntimeException("Recieved string of excessive length.");
			return "ERROR STRING RCVD";
		}
	}

	public static void SendPlayersData() throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(PLAYERS_DATA);
			Client.netClient.getOutput().writeFloat(Client.VERSION);
			WriteString(Client.netClient.getOutput(), Client.players.getThisPlayer().getName());
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodePlayersData(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;
		try {
			float client_version = conn.getInput().readFloat();
			String name = ReadString(conn.getInput());

			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding players data.");
			}

			System.out.println("Players version:" + client_version);
			if (client_version < Client.VERSION) {
				S2CCommunication.SendMessage(pdata, "There is a newer version of the software available.  Please upgrade as soon as possible.", (byte)2);
			} else if (client_version > Client.VERSION) {
				S2CCommunication.SendMessage(pdata, "WARNING!  The server you have connected to is running an older version of the software (" + Client.VERSION + ").", (byte)2);
				System.out.println("Warning - server software is too old for client.");
				System.out.println("Server: " + Client.VERSION);
				System.out.println("Client: " + client_version);
			}

			//S2CCommunication.SendMessage(conn, "You are connected to " + Server.getName() + ".", (byte)0);
			Server.players.setPlayersName(conn, name);

			S2CCommunication.SendSideOptions(pdata);
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendPlayersSide() throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(SIDE_SELECTED);
			Client.netClient.getOutput().writeByte(Client.players.getThisPlayer().side);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodePlayersSide(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;
		try {
			byte side = conn.getInput().readByte();
			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding players side.");
			}

			Server.players.setPlayersSide(conn, side);

			S2CCommunication.SendNewPlayerToExistingPlayers(pdata);
			S2CCommunication.SendExistingPlayersDetailsToNewPlayer(pdata);
			S2CCommunication.SendGameData(pdata);
			S2CCommunication.SendNewMap(pdata);

			// Send all objects
			SObject object;
			for (int p=0 ; p<Server.objects.length ; p++) {
				object = Server.objects[p];
				if (object != null) {
					S2CCommunication.SendNewObject(pdata, object);
					if (object instanceof SPlayerObject3D) {
						SPlayerObject3D obj = (SPlayerObject3D) object;
						S2CCommunication.SendTroopersItems(pdata, obj);
						S2CCommunication.SendHealthLevel(pdata, obj);
						S2CCommunication.SendArmourLevel(pdata, obj);
						S2CCommunication.SendTrooperStats(pdata, obj);
						S2CCommunication.SendCanSeeEnemy(obj);
						S2CCommunication.SendGuard(obj);
						S2CCommunication.SendAutofire(obj);
						if (obj.getCurrentController() != null) { // Send the details of who's controlling them
							S2CCommunication.SendSelectUnit(obj.getCurrentController(), obj);
						}
					}
				}
			}

			S2CCommunication.SendVPUpdate(side);
			S2CCommunication.SendElapsedTime();

			// Tell them if no-one is controlling the other side
			byte other_side = ServerFunctions.getOppositeSide(side);
			if (Server.game.isControlledByCPU(other_side)) {
				S2CCommunication.SendMessage(pdata, "You are playing against the CPU.", (byte)0);
			} else if (Server.game.getNoOfPlayersOnSide(other_side) == 0) {
				S2CCommunication.SendMessage(pdata, "No-one is currently controlling the opposite side.  You can either wait for others to join or play against the CPU.", (byte)1);
			}
			S2CCommunication.SendMessage(pdata, "There are " + Server.game.getNoOfUnitsOnSide(other_side) + " units remaining on the opposing side.", (byte)0);

			/*
			 * Put them in ghost mode if no space
			 */
			if (Server.game.getNoOfPlayersOnSide(side) <= Server.game.getMaxNoOfPlayersOnSide(side)) {
				pdata.selectNextUnit(); // Give them the next available unit
				S2CCommunication.SendMessage(pdata, "Press G for game options, and U for unit options.", (byte)0);
			} else {
				S2CCommunication.SendMessage(pdata, "There are no available units to select.", (byte)2);
				S2CCommunication.SendSetGhostMode(pdata);
				S2CCommunication.SendMessage(pdata, "Press G for game options.", (byte)0);
			}

		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}

	}

	public static void SendChangedItem(CPlayerObject3D trooper) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(CHANGED_ITEM);
			Client.netClient.getOutput().writeInt(trooper.id);
			if (trooper.getCurrentItem() != null) {
				Client.netClient.getOutput().writeInt(trooper.getCurrentItem().no);
			} else {
				Client.netClient.getOutput().writeInt(-1);
			}
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeChangedItem(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;
		try {
			int trooperno = conn.getInput().readInt();
			int itemno = conn.getInput().readInt();
			SPlayerObject3D trooper = (SPlayerObject3D) Server.objects[trooperno];
			if (trooper != null) {
				if (itemno != -1) {
					trooper.setCurrentItemByNo(itemno);
					trooper.reduceAPs(SPlayerObject3D.CHANGE_ITEM_COST, true);
				}
				else {
					trooper.setCurrentItemByNo(-1);
				}
			} else {
				System.err.println("Decode changed item pointed to null trooper.");
			}
			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding changed item.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendAutofire(CPlayerObject3D obj) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(AUTOFIRE);
			Client.netClient.getOutput().writeInt(obj.id);
			Client.netClient.getOutput().writeBoolean(obj.isAutofiring());
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeAutofire(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;
		try {
			int no = conn.getInput().readInt();
			boolean autofire = conn.getInput().readBoolean();
			SPlayerObject3D trooper = (SPlayerObject3D) Server.objects[no];
			if (trooper != null) {
				trooper.setAutofire(autofire, false);
			} else {
				System.err.println("Decode autofire pointed to null trooper.");
			}
			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding autofire.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendGuard(CPlayerObject3D obj) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(GUARD);
			Client.netClient.getOutput().writeInt(obj.id);
			Client.netClient.getOutput().writeBoolean(obj.isGuarding());
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeGuard(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;
		try {
			int no = conn.getInput().readInt();
			boolean guard = conn.getInput().readBoolean();
			SPlayerObject3D obj = (SPlayerObject3D) Server.objects[no];
			if (obj != null) {
				obj.setGuard(guard, false);
				if (obj.getAPs() >= Server.MAX_ACTION_POINTS) {
					obj.setAPs(Server.MAX_ACTION_POINTS - 1);
				}
			} else {
				System.err.println("Decode guard pointed to null trooper.");
			}
			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding guard.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendReload(CPlayerObject3D obj) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(RELOAD);
			Client.netClient.getOutput().writeInt(obj.id);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeReload(SPlayerData pdata){
		ClientConnection conn = pdata.conn;
		try {
			int no = conn.getInput().readInt();
			SPlayerObject3D obj = (SPlayerObject3D) Server.objects[no];
			if (obj != null) {
				obj.reload(false);
			} else {
				System.err.println("Decode reload pointed to null trooper.");
			}
			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding reload.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendAction(CPlayerObject3D obj) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(ACTION);
			Client.netClient.getOutput().writeInt(obj.id);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeAction(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;
		try {
			int no = conn.getInput().readInt();
			SPlayerObject3D obj = (SPlayerObject3D) Server.objects[no];
			if (obj != null) {
				obj.performActionIfEnoughAPs();
			} else {
				System.err.println("Decode action pointed to null trooper.");
			}
			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding action.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendUseItem(CPlayerObject3D obj) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(USE_ITEM);
			Client.netClient.getOutput().writeInt(obj.id);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeUseItem(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;
		try {
			int no = conn.getInput().readInt();
			SPlayerObject3D unit = (SPlayerObject3D) Server.objects[no];
			if (unit != null) {
				if (unit.getCurrentItem() != null) {
					unit.useCurrentItem();
				}
				else {
					System.err.println("Trying to use a null item!");
				}
			} else {
				System.err.println("Decode use item points to null trooper.");
			}
			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding autofire.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendPrimeAndThrow(CPlayerObject3D trooper, int primetime, int dist) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(PRIME_AND_THROW);
			Client.netClient.getOutput().writeInt(trooper.id);
			Client.netClient.getOutput().writeInt(trooper.getCurrentItem().no);
			Client.netClient.getOutput().writeInt(primetime);
			Client.netClient.getOutput().writeInt(dist);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodePrimeAndThrow(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;
		try {
			int trooperno = conn.getInput().readInt();
			int itemno = conn.getInput().readInt();
			int primetime = conn.getInput().readInt();
			int dist = conn.getInput().readInt();

			if (Client.SHOW_COMMS) {
				System.out.println("Throwing grenade.  Dist " + dist + ", Timer " + primetime + ".");
			}

			SPlayerObject3D trooper = (SPlayerObject3D) Server.objects[trooperno];
			if (trooper != null) {
				SGrenadeCarried gren = (SGrenadeCarried) trooper.getItemByRef(itemno);
				gren.primeAndThrow(trooper, primetime, dist);
			} else {
				System.err.println("Decode prime and throw pointed to null trooper.");
			}

			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding prime and throw.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendKeyPressed(byte code) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			DataOutputStream dos = Client.netClient.getOutput();
			dos.writeByte(KEY_PRESSED);
			dos.writeByte(code);
			dos.writeByte(CHECK);
			dos.flush();
		}
	}

	public static void DecodeKeyPressed(SPlayerData pdata) {
		try {
			byte code = pdata.conn.getInput().readByte();
			if (Client.SHOW_KEY_PRESSES) {
				System.out.println("Server: Key pressed: " + code);
			}
			pdata.keys[code] = true;
			byte check = pdata.conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding key pressed.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendKeyReleased(byte code) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			DataOutputStream dos = Client.netClient.getOutput();
			dos.writeByte(KEY_RELEASED);
			dos.writeByte(code);
			dos.writeByte(CHECK);
			dos.flush();
		}
	}

	public static void DecodeKeyReleased(SPlayerData pdata) {
		try {
			byte code = pdata.conn.getInput().readByte();
			if (Client.SHOW_KEY_PRESSES) {
				System.out.println("Server: Key released: " + code);
			}
			pdata.keys[code] = false;
			byte check = pdata.conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding key released.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendChangedTrooper(CPlayerObject3D trooper) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			DataOutputStream dos = Client.netClient.getOutput();
			dos.writeByte(CHANGED_TROOPER);
			dos.writeInt(trooper.id);
			dos.writeByte(CHECK);
			dos.flush();
		}
	}

	public static void DecodeChangedTrooper(SPlayerData pdata) {
		try {
			int trooperno = pdata.conn.getInput().readInt();
			pdata.selectUnit((SPlayerObject3D) Server.objects[trooperno]);
			byte check = pdata.conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding changed trooper.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendWait(CPlayerObject3D obj) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(WAIT);
			Client.netClient.getOutput().writeInt(obj.id);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeWait(SPlayerData pdata) {
		try {
			ClientConnection conn = pdata.conn;
			int no = conn.getInput().readInt();
			SPlayerObject3D trooper = (SPlayerObject3D) Server.objects[no];
			if (trooper != null) {
				trooper.reduceAPs(10, true);
				if (trooper.getAPs() < 0) {
					trooper.setAPs(0);
				}
			} else {
				System.err.println("Wait pointed to null trooper.");
			}
			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding wait.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendDisconnect() throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(DISCONNECT);
			//Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeDisconnect(SPlayerData pdata) {
		Server.players.removePlayer(pdata);
	}

	public static void SendShotType(CPlayerObject3D obj) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(SHOT_TYPE);
			Client.netClient.getOutput().writeInt(obj.id);
			Client.netClient.getOutput().writeInt(obj.getShotType());
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeShotType(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;
		try {
			int no = conn.getInput().readInt();
			int type = conn.getInput().readInt();
			SPlayerObject3D trooper = (SPlayerObject3D) Server.objects[no];
			if (trooper != null) {
				trooper.setShotType(type);
			} else {
				System.err.println("Decode shot_type pointed to null trooper.");
			}
			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding shot_type.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendAngle(CPlayerObject3D obj) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			DataOutputStream dos = Client.netClient.getOutput();
			dos.writeByte(ANGLE);
			dos.writeInt(obj.id);
			dos.writeInt(obj.y_angle);
			dos.writeByte(CHECK);
			dos.flush();
		}
	}

	public static void DecodeAngle(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;

		try {
			DataInputStream dis = conn.getInput();
			int no = dis.readInt();
			int ang = dis.readInt();
			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding angle.");
			}

			SPlayerObject3D obj = (SPlayerObject3D) Server.objects[no];
			if (obj != null) {
				float cost = Functions.GetDiffBetweenAngles(ang, obj.getAngle()) * Client.TURN_COST;
				obj.turnRegardless(ang, true, true, false); // We need to send comms so other players see the turning!
				obj.reduceAPs(cost, false);
			} else {
				System.err.println("Decode angle pointed to null trooper.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendUnitsLocation(CPlayerObject3D obj) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			DataOutputStream dos = Client.netClient.getOutput();
			dos.writeByte(LOCATION);
			dos.writeInt(obj.id);
			dos.writeFloat(obj.getX());
			dos.writeFloat(obj.getZ());
			dos.writeByte(CHECK);
			dos.flush();
		}
	}

	public static void DecodeUnitsLocation(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;

		try {
			DataInputStream dis = conn.getInput();
			int no = dis.readInt();
			float x = dis.readFloat();
			float z = dis.readFloat();
			byte check = conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error decoding location.");
			}

			SPlayerObject3D obj = (SPlayerObject3D) Server.objects[no];
			if (obj != null) {
				obj.setLocation(x, z, false);
				float cost = obj.move_fwd_cost * Client.MOVE_SPEED;
				cost = PlayerObject3D.adjustCostForArmour(obj.armour_damage_reduction, cost);

				obj.reduceAPs(cost, false);
			} else {
				System.err.println("Decode location pointed to null trooper.");
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendPlayersName(String name) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			if (Client.SHOW_COMMS) {
				System.out.println("Sending name");
			}
			DataOutputStream dos = Client.netClient.getOutput();
			dos.writeByte(PLAYERS_NAME);
			dos.writeInt(name.length());
			dos.write(name.getBytes());
			dos.writeByte(CHECK);
			dos.flush();
			if (Client.SHOW_COMMS) {
				System.out.println("Finished sending name");
			}
		}
	}

	public static void DecodePlayersName(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;
		if (Client.SHOW_COMMS) {
			System.out.println("Decoding name.");
		}
		DataInputStream dis = conn.getInput();
		try {
			int len = dis.readInt();
			byte b[] = new byte[len];
			while (dis.available() < len) {}
			dis.read(b);
			byte check = dis.readByte();
			if (check != CHECK) {
				System.err.println("Error on decode name.");
			}
			String name = new String(b);
			Server.players.setPlayersName(conn, name);
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendMessage(String msg, boolean own_side_only) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			if (Client.SHOW_COMMS) {
				System.out.println("Sending msg");
			}
			DataOutputStream dos = Client.netClient.getOutput();
			dos.writeByte(MSG);
			dos.writeBoolean(own_side_only);
			WriteString(dos, msg);
			dos.writeByte(CHECK);
			dos.flush();
			if (Client.SHOW_COMMS) {
				System.out.println("Finished sending msg");
			}
		}
	}

	public static void DecodeMessage(SPlayerData pdata) {
		ClientConnection conn = pdata.conn;
		if (Client.SHOW_COMMS) {
			System.out.println("Decoding msg.");
		}
		DataInputStream dis = conn.getInput();
		try {
			boolean own_side_only = dis.readBoolean();
			String msg = ReadString(dis);
			byte check = dis.readByte();
			if (check != CHECK) {
				System.err.println("Error on decode msg.");
			}
			String sender = Server.players.GetPlayersNameByConn(conn);
			byte side = Server.players.GetPlayersSideByConn(conn);

			Server.frm_log.log("Msg: " + pdata.getName() + ": " + msg);

			if (own_side_only) {
				S2CCommunication.SendMessageToAllOnSide(sender + ": " + msg, (byte)0, side);
			} else {
				S2CCommunication.SendMessageToAll(sender + ": " + msg, (byte)0);
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendPing() throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(PING);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
			Client.ping_test.pingSent();
		}
	}

	public static void DecodePing(SPlayerData pdata) {
		try {
			byte check = pdata.conn.getInput().readByte();
			if (check != CHECK) {
				System.err.println("Error on decode ping.");
			}
			S2CCommunication.SendPingReply(pdata);
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendSelectNextUnit() throws IOException {
		synchronized (Client.netClient.getOutput()) {
			if (Client.SHOW_COMMS) {
				System.out.println("Sending SelectNextUnit to server");
			}
			Client.netClient.getOutput().writeByte(SELECT_NEXT_UNIT);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeSelectNextUnit(SPlayerData pdata) {
		if (Client.SHOW_COMMS) {
			System.out.println("Decoding selectNextUnit from client");
		}
		DataInputStream dis = pdata.conn.getInput();
		try {
			byte check = dis.readByte();
			if (check != CHECK) {
				System.err.println("Error on decode select next unit.");
			}
			pdata.selectNextUnit();
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendSelectPrevUnit() throws IOException {
		synchronized (Client.netClient.getOutput()) {
			if (Client.SHOW_COMMS) {
				System.out.println("Sending SelectPrevUnit.");
			}
			Client.netClient.getOutput().writeByte(SELECT_PREV_UNIT);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeSelectPrevUnit(SPlayerData pdata) {
		if (Client.SHOW_COMMS) {
			System.out.println("Decoding SelectPrevUnit.");
		}
		DataInputStream dis = pdata.conn.getInput();
		try {
			byte check = dis.readByte();
			if (check != CHECK) {
				System.err.println("Error on decode select prev unit.");
			}

			pdata.selectPrevUnit();
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendDeselectCurrentUnit() throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(DESELECT_UNIT);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeDeselectCurrentUnit(SPlayerData pdata) {
		DataInputStream dis = pdata.conn.getInput();
		try {
			byte check = dis.readByte();
			if (check != CHECK) {
				System.err.println("Error on decode deselect unit.");
			}
			pdata.deselectCurrentUnit();
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendPlayAgainstCPU() throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(PLAY_AGAINST_CPU);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodePlayAgainstCPU(SPlayerData pdata) {
		DataInputStream dis = pdata.conn.getInput();
		try {
			byte check = dis.readByte();
			if (check != CHECK) {
				System.err.println("Error on decode ask play against CPU.");
			}

			boolean has_player[] = new boolean[2];
			has_player[Client.ATTACKER] = false;
			has_player[Client.DEFENDER] = false;

			for (int u=0 ; u < Server.objects.length ; u++) {
				if (Server.objects[u] != null) {
					if (Server.objects[u] instanceof SPlayerObject3D) {
						SPlayerObject3D unit = (SPlayerObject3D)Server.objects[u];
						if (unit.getCurrentController() != null || unit.isComputerControlled()) {
							has_player[Server.objects[u].side] = true;
						}
					}
				}
			}

			for (byte s=0 ; s<2 ; s++) {
				if (has_player[s] == false) {
					if (Server.game.mission.setupAI(s)) {
						S2CCommunication.SendMessageToAll("Enemy AI generated for the " + ServerFunctions.getSideName(s) + "s.", (byte)1);
						Server.game.setControlledByCPU(s, true);
					} else {
						S2CCommunication.SendMessageToAll("Could not generate Enemy AI for the " + ServerFunctions.getSideName(s) + "s.", (byte)1);
					}
				} else {
					//S2CCommunication.SendMessageToAll("There are already controllers for the " + ServerFunctions.getSideName(s) + "s.", (byte)0);
				}
			}
		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}

	}

	public static void SendObjectsCollided(CObject obj1, CObject obj2) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(OBJECTS_COLLIDED);
			Client.netClient.getOutput().writeInt(obj1.id);
			Client.netClient.getOutput().writeInt(obj2.id);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeObjectsCollided(SPlayerData pdata) {
		DataInputStream dis = pdata.conn.getInput();
		try {
			int id1 = dis.readInt();
			int id2 = dis.readInt();
			byte check = dis.readByte();
			if (check != CHECK) {
				System.err.println("Error on decode objects collided.");
			}

			SObject obj1 = Server.objects[id1];
			SObject obj2 = Server.objects[id2];
			obj1.hasCollidedWith(obj2);

		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

	public static void SendVote(boolean y) throws IOException {
		synchronized (Client.netClient.getOutput()) {
			Client.netClient.getOutput().writeByte(VOTE_SENT);
			Client.netClient.getOutput().writeBoolean(y);
			Client.netClient.getOutput().writeByte(CHECK);
			Client.netClient.getOutput().flush();
		}
	}

	public static void DecodeVoteSent(SPlayerData pdata) {
		DataInputStream dis = pdata.conn.getInput();
		try {
			boolean b = dis.readBoolean();
			byte check = dis.readByte();
			if (check != CHECK) {
				System.err.println("Error on decode vote sent.");
			}

			//todo!

		} catch (IOException e) {
			Server.HandleError(e);
			Server.players.removePlayer(pdata);
		}
	}

}
