package ngrave.ai;

import java.awt.Point;
import java.io.IOException;

import ngrave.client.Client;
import ngrave.server.CollisionDetails;
import ngrave.server.SMapSquare;
import ngrave.server.Server;
import ngrave.server.ServerFunctions;
import ngrave.server.items.SWeaponCarried;
import ngrave.server.objects.SGrenade;
import ngrave.server.objects.SObject;
import ngrave.server.objects.SPlayerObject3D;
import ssmith.astar.AStar;
import ssmith.astar.IAStarMapInterface;
import ssmith.astar.WayPoints;
import ssmith.lang.Functions;
import ssmith.util.Interval;

public abstract class AI implements IAStarMapInterface {

	private static final long MOVE_PATIENCE = 60;

	protected final SPlayerObject3D trooper;
	protected SPlayerObject3D enemy;
	private AStar astar;
	protected WayPoints route = new WayPoints();
	private Point untraversable;  // Holds temp points that are untraversable
	private Interval check_environment_interval = new Interval(1000);
	private SGrenade avoid_grenade = null; // The grenade we're avoiding
	private boolean check_lr_at_next_square = false, started_checking_lr = false, checked_right=false, checked_left=false;
	private int target_right_angle=0, target_left_angle = 0;
	private long unsuccessful_move_count = 0;

	public AI(SPlayerObject3D po) {
		this.trooper = po;
		this.trooper.setAPs(0); // So they don't spend the first 5 mins running round at the speed of light using up all their points.
	}

	protected abstract void performNonCombatAI();

	public void doAI() {
		//System.out.println("Doing AI.");

		if (trooper.getAPs() <= 1) {
			return;
		}

		if (this.astar != null) {
			if (astar.finishedFindingPath() == false) {
				return;
			} else {
				route = astar.getRoute();
				astar = null;
			}
		}

		// Are we avoiding a grenade?
		if (this.avoidingGrenade()) {
			// Has it exploded?
			if (avoid_grenade.isDestroyed() == true) {
				avoid_grenade = null;
			} else {
				// Keep moving!
				if (route.hasAnotherPoint()) {
					this.moveUsingAStar();
					return;
				} else {
					// Do nothing - wait till its exploded
					this.trooper.ensureNoHoldup();
					return;
				}
			}
		} else {
			if (check_environment_interval.hitInterval()) {
				if (this.checkEnvironment()) {
					return;
				}
			}
		}

		// Check we've got enough AP's to shoot.
		if (trooper.getCurrentItem() instanceof SWeaponCarried) {
			SWeaponCarried wep = (SWeaponCarried) trooper.getCurrentItem();
			if (wep != null) {
				if (trooper.getAPs() <= wep.shot_aps) {
					// Don't do anything - we can't shoot anyway.
					//System.out.println("Not enough APs (" + trooper.getAPs() + ")");
					return;
				}
			} else {
			    System.out.println("Warning - trooper " + trooper.toString() + " does not have a weapon.");
			}
		}

		if (enemy != null) {
			if (trooper.canSee(enemy, true)) {
				if (Client.SHOW_AI) {
					System.out.println("AI " + trooper.name + " shooting!");
				}
				trooper.turnRegardless(trooper.getAbsoluteAngleTo(enemy), true, true, false); // Point to the target
				if (trooper.isAngleClearOfComrades(trooper.distanceTo(enemy))) {
					trooper.reload(true); // Just in case
					trooper.shoot();
				} else {
					// Angle not clear!
					if (Client.SHOW_AI) {
						System.out.println("Angle not clear for a shot!");
					}
					if (Functions.rnd(0, 1) == 1) {
						trooper.slideLeftIfEnoughAPs();
					} else {
						trooper.slideRightIfEnoughAPs();
					}
					// Check we're not holding the game up
					trooper.ensureNoHoldup();
				}
				return;
			} else {
				// Can't see them any more.
				enemy = null;
			}
		}

		if (trooper.getAPs() >= Server.MAX_ACTION_POINTS/2) { // Don't do anything unless we've got loads of points.
			if (check_lr_at_next_square == false) {
				// Are we walking through a door?
				SMapSquare sq = trooper.getMapSquare();
				if (sq.getType() == SMapSquare.OPEN_DOOR || sq.getType() == SMapSquare.CLOSED_DOOR) {
					check_lr_at_next_square = true;
					started_checking_lr = false;
					target_left_angle = Functions.RestateAngle(trooper.getAngle() + 90);
					target_right_angle = Functions.RestateAngle(trooper.getAngle() - 90);
				}
			}
			if (check_lr_at_next_square) { // Is either side viewable
				if (started_checking_lr == false) {
					// Have we got to through the door yet?
					if (trooper.isInMiddleOfSquare()) {
						SMapSquare sq = trooper.getMapSquare();
						if (sq.getType() != SMapSquare.OPEN_DOOR && sq.getType() != SMapSquare.CLOSED_DOOR) {
							if (this.trooper.getMapSquareAtAbsoluteAngle(target_right_angle).blocks_targetting_view == false) {
								checked_right = false;
							} else {
								checked_right = true;
							}
							if (this.trooper.getMapSquareAtAbsoluteAngle(target_left_angle).blocks_targetting_view == false) {
								checked_left = false;
							} else {
								checked_left = true;
							}
							started_checking_lr = !checked_right || !checked_left;
						}
					}
				}
				if (started_checking_lr) {
					// We're through the door, so start looking left/right.
					if (checked_right == false) {
						this.trooper.turnTowardsIfEnoughAPs(target_right_angle);
						if (this.trooper.isWithinAngle(target_right_angle, 10)) {
							checked_right = true;
						}
					} else if (checked_left == false) {
						this.trooper.turnTowardsIfEnoughAPs(target_left_angle);
						if (this.trooper.isWithinAngle(target_left_angle, 10)) {
							checked_left = true;
						}
					}
					if (checked_right == true && checked_left == true) {
						this.check_lr_at_next_square = false;
					} else {
						return;// Keep turning next time.
					}
				}
			}
			enemy = trooper.findAnEnemy();
			if (enemy == null) {
				performNonCombatAI();
			}
		}
	}

	public boolean avoidingGrenade() {
		return this.avoid_grenade != null;
	}

	/**
	 * This returns whether to continue checking AI stuff.
	 * @return
	 * @throws IOException
	 */
	private boolean checkEnvironment() {
		SObject obj;
		for (int o = 0; o < Server.objects.length; o++) {
			obj = Server.objects[o];
			if (obj instanceof SGrenade) {
				SGrenade gren = (SGrenade)obj;
				int dist = (int)trooper.distanceTo(gren);
				if (dist <= ServerFunctions.EXPLOSION_RANGE) {
					if (trooper.canSee(gren, true)) {
						avoid_grenade = gren;
						// Get a route away from grenade.  Find safe location first
						GetRouteToSafeLocation(gren, ServerFunctions.EXPLOSION_RANGE);
						return true;
					}
				}
			}
		}
		return false;
	}

	/**
	 * This finds the closest safe location from something.
	 * @param trooper
	 * @param grenade
	 * @param expl_radius
	 * @return
	 */
	private void GetRouteToSafeLocation(SObject grenade, int expl_radius) {
		int max_z = 0, max_x = 0;
		int start_x = (int)trooper.getX();
		int start_z = (int)trooper.getZ();
		System.out.println("Starting from " + start_x + "," + start_z);
		System.out.println("Avoiding from " + (int)grenade.getX() + "," + (int)grenade.getZ());
		while (true) {
			max_x++;
			max_z++;
			if (max_x > expl_radius*2) {
				return;
			}
			for (int z = start_z-max_z ; z<=start_z+max_z ; z++) {
				for (int x = start_x-max_x ; x<=start_x+max_x ; x++) {
					if (x == start_x-max_x || x == start_x+max_x || z == start_z-max_z || z == start_z+max_z) {
						try {
							//System.out.println("Trying square " + x + "," + z);
							// Can we walk on it?
							if (trooper.canTraverse(Server.map.getMapSquare(x, z))) {
								// Is it far enough away from the grenade?  or is it out of sight of the grenade?
								int dist_away = (int)trooper.distanceTo(Server.map.getMapSquare(x, z));
								boolean out_of_sight = ServerFunctions.CanSee(grenade.getX(), grenade.getZ(), x+0.5f, z+0.5f) == false;
								if (dist_away > expl_radius || out_of_sight) {
									// Can we walk TO it?
									//System.out.println("Trying route from " + start_x + "," + start_z + " to " + x + "," + z);
									trooper.calcRoute(x, z, dist_away*2);
								}
							}
						} catch (java.lang.IndexOutOfBoundsException ex) {
							// Probably caused by an object being removed from Objects by another thread
							// System.err.println(ex.getMessage());
							//ex.printStackTrace();
						}
					}
				}
			}
		}
	}

	/**
	 * Override this to do something else.
	 * @param shooter
	 * @throws IOException
	 */
	public void hitByBullet(SPlayerObject3D shooter) {
		// Only do stuff if its not on our side.
		if (shooter.side != trooper.side) {
			// Face enemy.
			boolean result = trooper.turnIfEnoughAPs(trooper.getAbsoluteAngleTo(shooter), true); // Point to the target
			if (Client.SHOW_AI) {
				if (result) {
					System.out.println("AI "+trooper.name + " hit by bullet - turning to " + shooter.name + ".");
				} else {
					System.out.println("AI "+trooper.name + " hit by bullet BUT NOT ENOUGH APS to turn to " + shooter.name + ".");
				}
			}
			trooper.checkForEnemyNextTime();
		}
	}

	//-- A* stuff ------------------------------------

	public void calcRoute(Point p, int max_dist) {
		calcRoute(p.x, p.y, max_dist);
	}

	public void calcRoute(int x, int z, int max_dist) {
		//Server.log("Thinking for " + trooper.name + "...");
		//S2CCommunication.SendMessageToAll(this.trooper.name +" is thinking...", (byte)0);
		astar = new AStar(this);
		astar.findPath((int)trooper.getX(), (int)trooper.getZ(), x, z, max_dist, true);
		//Server.log("Finished.");
		//S2CCommunication.SendMessageToAll(this.trooper.name +" is ready.", (byte)0);
		//return route.hasAnotherPoint();
	}

	/**
	 * This will adjust the route by marking the next point
	 * as untraversable, and seeing how far they can get.
	 *
	 */
	public void adjustRoute() {
		//System.out.println("Adjusting route for " + trooper.name + ".");
		//S2CCommunication.SendMessageToAll(this.trooper.name +" is thinking...", (byte)0);
		untraversable = (Point)route.getNextPoint();
		Point dest = (Point)route.getLastPoint();
		astar = new AStar(this);
		astar.findPath((int)trooper.getX(), (int)trooper.getZ(), dest.x, dest.y, false); // needs to be false to avoid java.lang.IllegalThreadStateException
		untraversable = null;
		//S2CCommunication.SendMessageToAll(this.trooper.name +" is ready.", (byte)0);
		//System.out.println("Finished adjusting route.");
	}

	public int getMapWidth() {
		return Server.map.size;
	}

	public int getMapHeight() {
		return Server.map.size;
	}

	public boolean isMapSquareTraversable(int x, int z) {
		if (untraversable != null) {
			if (untraversable.x == x && untraversable.y == z) {
				return false;
			}
		}
		SMapSquare sq = Server.map.getMapSquare(x, z);
		if (sq.getType() == SMapSquare.CLOSED_DOOR) {
			return true;
		} else {
			return this.trooper.canTraverse(sq);
		}
	}

	protected void moveUsingAStar() {
		if (route.hasAnotherPoint()) {
			Point next_step = route.getNextPoint();
			int angle_to_ms = this.trooper.getAbsoluteAngleTo(Server.map.getMapSquare(next_step.x, next_step.y));
			this.trooper.turnTowardsIfEnoughAPs(angle_to_ms);
			if (this.trooper.isWithinAngle(angle_to_ms, Client.TURN_SPEED)) {
				if (trooper.moveForwardsIfEnoughAPs().moveSuccessful() == false) {
					CollisionDetails nmr = trooper.getNoMoveReason();
					if (nmr.blockedByScenery()) {
						// Is it a door?
						SMapSquare sq = trooper.getFacingMapSquare();
						if (trooper.canTraverse(sq) == false) {
							if (sq.getType() == SMapSquare.CLOSED_DOOR) {
								Server.map.openDoor(sq);
							} else {
								System.err.println("Error - cannot move in direction of A* (ms=" + sq.getType() + ")");
								route.clear(); // Ready to get another route.
							}
						} else {
							this.adjustRoute();
						}
					} else if (nmr.blockedByObject()) {
						// See if it's another trooper moving in our direction
						if (nmr.getBlockingObject() instanceof SPlayerObject3D) {
							SPlayerObject3D blocker = (SPlayerObject3D) nmr.getBlockingObject();
							if (Functions.mod(blocker.getAngle() - this.trooper.getAngle()) > 90) {
								this.trooper.moveBackwardsIfEnoughAPs();
								this.adjustRoute();
							} else { // Wait for them
								this.trooper.moveBackwardsIfEnoughAPs();
							}
						} else {
							//System.err.println("Trooper " + trooper.name + " is blocked by another trooper.");
							this.trooper.moveBackwardsIfEnoughAPs();
							this.trooper.slideRightIfEnoughAPs();
							this.adjustRoute();
						}
					} else {
						System.err.println("Unknown reason for no movement for " + trooper.name + ".");
					}
					unsuccessful_move_count++;
					if (unsuccessful_move_count > MOVE_PATIENCE) {
						System.out.println("** Trooper " + this.trooper.name + " reverted to wandering.");
						this.trooper.setAI(new Wander90AI(this.trooper));
					}
				} else {
					unsuccessful_move_count = 0;
				}
				if ((int)(this.trooper.getX()) == next_step.x && (int)(this.trooper.getZ()) == next_step.y) {
					if (trooper.isInMiddleOfSquare()) {
						route.removeCurrentPoint();
					}
				}
			}
		}
	}

	public float getMapSquareDifficulty(int x, int z) {
		return 1;
	}

}

